<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Admin
 */

namespace Application\Admin\Tests\Routes;


use Application\Admin\AdminSidebar\Sidebar;
use Application\Admin\Routes\PonghoRoute;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Request;

class PonghoRouteTest extends \PHPUnit_Framework_TestCase
{
    public function testMatchWithUserNotLogged()
    {
        $sidebar = $this->getMockBuilder('Application\Admin\AdminSidebar\Sidebar')
            ->disableOriginalConstructor()
            ->getMock();

        $user = $this->getMockBuilder('Application\Core\User')
            ->disableOriginalConstructor()
            ->getMock();

        $site = $this->getMockBuilder('Application\Core\Model\Site')
            ->disableOriginalConstructor()
            ->getMock();

        $site->expects($this->any())->method('getDomain')->willReturn('http://www.example.com');

        $pongho_url = $this->getMockBuilder('Application\Admin\Utilities\PonghoUrl')
            ->disableOriginalConstructor()
            ->getMock();

        $pongho_url->expects($this->any())->method('currentUrl')->with('&')->willReturn('/pongho/?foo=bar');

        $request = Request::create('/pongho/');

        /**
         * @var \Application\Admin\AdminSidebar\Sidebar $sidebar
         * @var \Application\Core\User                  $user
         * @var \Application\Core\Model\Site            $site
         * @var \Application\Admin\Utilities\PonghoUrl  $pongho_url
         */

        $route = new PonghoRoute($sidebar, $user, $site, $pongho_url);
        $response = $route->match($request);

        $this->assertInstanceOf('Pongho\Http\RedirectResponse', $response);

        /** @var \Pongho\Http\RedirectResponse $response */
        $this->assertEquals(
            'http://www.example.com/user/login/?redirect=%2Fpongho%2F%3Ffoo%3Dbar',
            $response->getRedirectUri()
        );
    }

    public function routeProvider()
    {
        return array(
            array('/pongho/', true, false),
            array('/pongho/path/', true, false),
            array('/pongho/path/a/', true, false),
            array('/pongho/path/c/', true, false),
            array('/pongho/pippo/', true, true),
            array('/example/', false, false),
        );
    }

    /**
     * @dataProvider routeProvider
     * @param $uri
     * @param $is_pongho
     * @param $throw_exception
     */
    public function testMatch($uri, $is_pongho, $throw_exception)
    {
        // Controlli per la correttezza della configurazione del test
        if ((!$is_pongho && $throw_exception) || ($is_pongho && substr($uri, 0, 7) !== '/pongho')) {
            $this->fail('The PonghoRoute could not handle non-Pongho urls, check your test case');
        }

        $sidebar = $this->getSidebar($uri);
        $sidebar->addPanel('Panel A', 'panela', '', 0);
        $sidebar->addItem('item', '/', 'panela', 10, 'Controller');

        $sidebar->addPanel('Panel B', 'panelb', '', 10);
        $sidebar->addItem('item 0', '/path/', 'panelb', 10, 'Controller');
        $sidebar->addItem('item 1', '/path/a/', 'panelb', 20, 'Controller');
        $sidebar->addItem('item 2', '/path/b/', 'panelb', 30, 'Controller');
        $sidebar->addItem('item 3', '/path-other/', 'panelb', 40, 'Controller');

        $sidebar->addRoute('/path/c/', 'Controller');

        $route = new PonghoRoute($sidebar, $this->getUserMock(true), $this->getSiteMock(), $this->getPonghoUrlMock(null));

        try {
            $match = $route->match($this->getRequestMock($uri));

            if ($throw_exception) {
                throw new \Exception();
            }

            $this->assertEquals($is_pongho, $match);
        } catch (HttpNotFoundException $e) {
            if (!$throw_exception) {
                $this->fail('Oops 404 not found when it should be found!');
            }
        } catch (\Exception $e) {
            $this->fail('An exception should have been thrown');
        }
    }

    /**
     * @param      $uri
     * @param null $domain
     * @return \Application\Admin\AdminSidebar\Sidebar
     */
    protected function getSidebar($uri, $domain = null)
    {
        return new Sidebar(
            $this->getSiteMock($domain),
            $this->getRequestMock($uri),
            $this->getPonghoUrlMock($domain)
        );
    }

    /**
     * @param null $domain
     * @return \PHPUnit_Framework_MockObject_MockObject|\Application\Core\Model\Site
     */
    protected function getSiteMock($domain = null)
    {
        $site = $this->getMockBuilder('Application\\Core\\Model\\Site')
            ->disableOriginalConstructor()
            ->getMock();

        $site->expects($this->any())
            ->method('domain')
            ->will($this->returnValue($domain));

        return $site;
    }

    /**
     * @param $uri
     * @return \PHPUnit_Framework_MockObject_MockObject|Request
     */
    protected function getRequestMock($uri)
    {
        return Request::create($uri);
    }

    /**
     * @param bool $logged
     * @return \PHPUnit_Framework_MockObject_MockObject|\Application\Core\User
     */
    protected function getUserMock($logged = true)
    {
        $user = $this->getMockBuilder('Application\\Core\\User')
            ->disableOriginalConstructor()
            ->getMock();

        $user->expects($this->any())->method('isLogged')->willReturn($logged);

        return $user;
    }

    /**
     * @param $domain
     * @return \PHPUnit_Framework_MockObject_MockObject|\Application\Admin\Utilities\PonghoUrl
     */
    protected function getPonghoUrlMock($domain)
    {
        $pongho_url = $this->getMockBuilder('Application\\Admin\\Utilities\\PonghoUrl')
            ->disableOriginalConstructor()
            ->getMock();

        if ($domain) {
            $pongho_url
                ->expects($this->any())
                ->method('url')
                ->will($this->returnCallback(function ($path) use ($domain) {
                    return $domain . $path . '?foo=bar';
                }));
        } else {
            $pongho_url
                ->expects($this->any())
                ->method('url')
                ->will($this->returnCallback(function ($path) {
                    return $path . '?foo=bar';
                }));
        }

        return $pongho_url;
    }
}
