<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Admin
 */

namespace Application\Admin\Tests\Panels;

use Application\Admin\Panels\PanelsManager;
use Symfony\Component\DomCrawler\Crawler;

class PanelsManagerTest extends \PHPUnit_Framework_TestCase
{
    public function testCollection()
    {
        $manager = new PanelsManager();
        $this->assertFalse($manager->has('foo'));
        $this->assertEquals(array(), $manager->keys());

        $panel = $this->getPanelMock('foo');

        $manager->add($panel);
        $this->assertTrue($manager->has('foo'));
        $this->assertSame($panel, $manager->get('foo'));
        $this->assertEquals(array('foo'), $manager->keys());

        $manager->del('foo');
        $this->assertFalse($manager->has('foo'));
        $this->assertEquals(array(), $manager->keys());
    }

    /**
     * @expectedException \LogicException
     */
    public function testAddConstrainsStringKey()
    {
        $manager = new PanelsManager();
        $manager->add($this->getPanelMock(null));
    }

    public function testAddFirst()
    {
        $manager = new PanelsManager();

        $manager->add($this->getPanelMock('foo'));
        $manager->add($this->getPanelMock('bar'));
        $manager->addFirst($this->getPanelMock('baz'));

        $this->assertEquals(array('baz', 'foo', 'bar'), $manager->keys());
    }

    /**
     * @expectedException \LogicException
     */
    public function testAddFirstConstrainsStringKey()
    {
        $manager = new PanelsManager();
        $manager->addFirst($this->getPanelMock(null));
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testNotFoundException()
    {
        $manager = new PanelsManager();
        $manager->get('foo');
    }

    public function testTraversable()
    {
        $manager = new PanelsManager();
        $this->assertInstanceOf('\Traversable', $manager);

        $manager->add($this->getPanelMock('foo'));
        $manager->add($this->getPanelMock('bar'));
        $this->assertEquals(array('foo', 'bar'), $manager->keys());

        $counter = 0;
        foreach ($manager as $panel) {
            $this->assertInstanceOf('Application\\Admin\\Panels\\PanelInterface', $panel);
            $counter++;
        }

        $this->assertEquals(2, $counter);
    }

    public function testRender()
    {
        $manager = new PanelsManager();

        $manager->add($this->getPanelForRenderMock('foo', 'Foo Title', 'Foo Content'));
        $manager->add($this->getPanelForRenderMock('bar', 'Bar Title', 'Bar Content'));
        $manager->add($this->getPanelForRenderMock('baz', 'Baz Title', 'Baz Content'));

        $crawler = new Crawler($manager->render());

        $this->assertEquals(3, $crawler->filter('ul.tabs li a')->count());
        $this->assertEquals(1, $crawler->filter('ul.tabs li a[href="#panel-foo"]')->count());
        $this->assertEquals(1, $crawler->filter('ul.tabs li a[href="#panel-bar"]')->count());
        $this->assertEquals(1, $crawler->filter('ul.tabs li a[href="#panel-baz"]')->count());

        $this->assertEquals(1, $crawler->filter('div.panels')->count());
        $this->assertEquals(3, $crawler->filter('div.panel')->count());
        $this->assertEquals(1, $crawler->filter('div.panels #panel-foo')->count());
        $this->assertEquals(1, $crawler->filter('div.panels #panel-bar')->count());
        $this->assertEquals(1, $crawler->filter('div.panels #panel-baz')->count());
    }

    /**
     * @dataProvider externalLinkProvider
     * @param string $url
     * @param array  $paths
     */
    public function testRenderWithExternalLink($url, $paths)
    {
        $pongho_url = $this->getMockBuilder('Application\\Admin\\Utilities\\PonghoUrl')
            ->disableOriginalConstructor()
            ->getMock();

        $pongho_url
            ->expects($this->any())
            ->method('url')
            ->willReturnCallback(
                function ($url) {
                    return $url;
                }
            );

        /**
         * @var \Application\Admin\Utilities\PonghoUrl $pongho_url
         */

        $manager = new PanelsManager(true, $url, $pongho_url);

        $manager->add($this->getPanelForRenderWithExternalLinkMock('foo', 'Foo Title', 'Foo Content', true));
        $manager->add($this->getPanelForRenderWithExternalLinkMock('bar', 'Bar Title', 'Bar Content', false));
        $manager->add($this->getPanelForRenderWithExternalLinkMock('baz', 'Baz Title', 'Baz Content', false));

        $crawler = new Crawler($manager->render());

        $tabs = $crawler->filter('ul.tabs.nojs');
        $this->assertEquals(1, $tabs->count());

        $tabs_links = $tabs->filter('li a');
        $this->assertEquals(3, $tabs_links->count());

        /** @var \DOMElement $link */
        $links = array();
        foreach ($tabs_links as $link) {
            $links[] = $link->getAttribute('href');
        }

        $this->assertEquals($paths, $links);

        $this->assertEquals(1, $crawler->filter('div.panels')->count());
        $this->assertEquals(1, $crawler->filter('div.panel')->count());
        $this->assertEquals(1, $crawler->filter('div.panels #panel-foo')->count());
        $this->assertEquals(0, $crawler->filter('div.panels #panel-bar')->count());
        $this->assertEquals(0, $crawler->filter('div.panels #panel-baz')->count());
    }

    public function externalLinkProvider()
    {
        return array(
            array('/foo', array('/foo/', '/foo/bar/', '/foo/baz/')),
            array('/foo/', array('/foo/', '/foo/bar/', '/foo/baz/')),
        );
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testExternalLinkWithoutRequestThrowException()
    {
        new PanelsManager(true);
    }

    public function testFirstPanelIsActive()
    {
        $manager = new PanelsManager();

        $manager->add($this->getPanelForRenderMock('foo', 'Foo Title', 'Foo Content'));
        $manager->add($this->getPanelForRenderMock('bar', 'Bar Title', 'Bar Content'));
        $manager->add($this->getPanelForRenderMock('baz', 'Baz Title', 'Baz Content'));

        $crawler = new Crawler($manager->render());

        $this->assertSame(
            $crawler->filter('ul.tabs li.active a')->text(),
            $crawler->filter('ul.tabs li')->first()->filter('a')->text()
        );
    }

    public function testSetActivePanel()
    {
        $manager = new PanelsManager();

        $manager->add($this->getPanelForRenderMock('foo', 'Foo Title', 'Foo Content'));
        $manager->add($this->getPanelForRenderMock('bar', 'Bar Title', 'Bar Content'));
        $manager->add($this->getPanelForRenderMock('baz', 'Baz Title', 'Baz Content'));

        $manager->setActive('bar');

        $crawler = new Crawler($manager->render());

        $this->assertSame(
            $crawler->filter('ul.tabs li.active a')->text(),
            $crawler->filter('ul.tabs li')->eq(1)->filter('a')->text()
        );
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testExceptionWhenSetActiveAPanelThatNotExists()
    {
        $manager = new PanelsManager();
        $manager->setActive('foo');
    }

    /**
     * @param string $key
     * @return \Application\Admin\Panels\PanelInterface
     */
    private function getPanelMock($key)
    {
        $panel = $this->getMock('Application\\Admin\\Panels\\PanelInterface');

        $panel->expects($this->once())->method('getKey')->willReturn($key);

        return $panel;
    }

    /**
     * @param string $key
     * @param string $title
     * @param string $content
     * @return \Application\Admin\Panels\PanelInterface
     */
    private function getPanelForRenderMock($key, $title, $content)
    {
        $panel = $this->getMock('Application\\Admin\\Panels\\PanelInterface');

        $panel->expects($this->any())->method('getKey')->willReturn($key);
        $panel->expects($this->once())->method('getTitle')->willReturn($title);
        $panel->expects($this->once())->method('getContent')->willReturn($content);

        return $panel;
    }

    /**
     * @param string $key
     * @param string $title
     * @param string $content
     * @param bool   $is_active
     * @return \Application\Admin\Panels\PanelInterface
     */
    private function getPanelForRenderWithExternalLinkMock($key, $title, $content, $is_active)
    {
        $panel = $this->getMock('Application\\Admin\\Panels\\PanelInterface');

        $panel->expects($this->any())->method('getKey')->willReturn($key);
        $panel->expects($this->once())->method('getTitle')->willReturn($title);

        if ($is_active) {
            $panel->expects($this->once())->method('getContent')->willReturn($content);
        } else {
            $panel->expects($this->never())->method('getContent')->willReturn($content);
        }

        return $panel;
    }
}
