<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Admin
 */

namespace Application\Admin\Form;

use Pongho\Core\LocalizationInterface;
use Pongho\Form\Subject\SubjectInterface;
use Pongho\Form\Utilities\FormConfig as BaseConfig;

class FormConfig extends BaseConfig
{
    /**
     * @var \Pongho\Core\LocalizationInterface
     */
    protected $lang;

    /**
     * @param string                                $name
     * @param \Pongho\Form\Subject\SubjectInterface $subject
     * @param \Pongho\Core\LocalizationInterface    $lang
     */
    public function __construct($name = '', SubjectInterface $subject = null, LocalizationInterface $lang)
    {
        parent::__construct($name, $subject);

        $this->lang = $lang;
    }


    /**
     * Crea la struttura di base delle Form di amministrazione
     *
     * @param string $active_tab
     * @param string $tab_url
     */
    public function addBaseStructure($active_tab, $tab_url)
    {
        $this->addFields(
            '',
            array(
                'content' => array(
                    'class' => 'Application\\Admin\\Form\\PanelsManagerFieldset',
                    'settings' => array(
                        'tab_active' => $active_tab,
                        'tab_url'    => $tab_url,
                    ),
                ),
                'actions' => array(
                    'class'      => 'Application\\Admin\\Form\\ActionsFieldset',
                    'attributes' => array('class' => 'footer-actions'),
                ),
            )
        );

        $this->addField(
            'actions/submit',
            array(
                'class' => 'Application\\Admin\\Form\\Field\\ActionsField',
                'label' => $this->lang->get('save'),
                'attributes' => array('class' => 'primary'),
                'settings'	=> array('type' => 'submit'),
            )
        );
    }

    /**
     * Crea la struttura con sidebar delle Form di amministrazione
     */
    public function addSidebarStructure()
    {
        $this->addBaseStructure('wrapper', '');
        $this->addTab('content/wrapper', '');

        $this->addFields(
            'content/wrapper',
            array(
                'container' => array(
                    'class' => 'Application\\Admin\\Form\\ContainerFieldset',
                ),
                'sidebar' => array(
                    'class'      => 'Application\\Admin\\Form\\ActionsFieldset',
                    'attributes' => array(
                        'class' => 'sidebar'
                    ),
                ),
            )
        );
    }

    /**
     * Funzione di supporto per addPanel*()
     *
     * @param null  $label
     * @param bool  $is_accordion
     * @param array $settings
     * @return array
     */
    protected function panel($label = null, $is_accordion = false, array $settings = array())
    {
        // L'accordion necessita sempre una label visibile, mentre se non specifico una label la nascondo di default
        if ($label === null && !$is_accordion) {
            $settings = array_replace_recursive(array('hide_label' => true), $settings);
        }

        $description = '';
        if (array_key_exists('description', $settings)) {
            $description = $settings['description'];
        }

        return array(
            'class'      => 'Pongho\\Form\\Fieldset',
            'label'      => $label,
            'attributes' => array('class' => 'panel' . ($is_accordion ? ' panel-accordion' : '')),
            'settings'   => $settings,
            'description' => $description,
        );
    }

    /**
     * Aggiunge un pannello al percorso specificato
     *
     * @param             $path
     * @param null|string $label
     * @param bool        $is_accordion
     * @param array       $settings
     */
    public function addPanel($path, $label = null, $is_accordion = false, array $settings = array())
    {
        $this->addField($path, $this->panel($label, $is_accordion, $settings));
    }

    /**
     * Aggiunge un pannello prima
     *
     * @param       $path
     * @param       $field
     * @param null  $label
     * @param bool  $is_accordion
     * @param array $settings
     */
    public function addPanelBefore($path, $field, $label = null, $is_accordion = false, array $settings = array())
    {
        $this->addFieldBefore($path, $field, $this->panel($label, $is_accordion, $settings));
    }

    /**
     * Aggiunge un pannello dopo
     *
     * @param       $path
     * @param       $field
     * @param null  $label
     * @param bool  $is_accordion
     * @param array $settings
     */
    public function addPanelAfter($path, $field, $label = null, $is_accordion = false, array $settings = array())
    {
        $this->addFieldAfter($path, $field, $this->panel($label, $is_accordion, $settings));
    }

    /**
     * Funzione di supporto per addTab*()
     *
     * @param       $label
     * @param array $settings
     * @return array
     */
    protected function tab($label, array $settings = array())
    {
        return array(
            'class' => 'Application\\Admin\\Form\\PanelFieldset',
            'label' => $label,
            'settings' => $settings,
        );
    }

    /**
     * Aggiunge una tab
     *
     * @param       $path
     * @param       $label
     * @param array $settings
     */
    public function addTab($path, $label, array $settings = array())
    {
        $this->addField($path, $this->tab($label, $settings));
    }

    /**
     * Aggiunge una tab prima
     *
     * @param       $path
     * @param       $field
     * @param       $label
     * @param array $settings
     */
    public function addTabBefore($path, $field, $label, array $settings = array())
    {
        $this->addFieldBefore($path, $field, $this->tab($label, $settings));
    }

    /**
     * Aggiunge una tab dopo
     *
     * @param       $path
     * @param       $field
     * @param       $label
     * @param array $settings
     */
    public function addTabAfter($path, $field, $label, array $settings = array())
    {
        $this->addFieldAfter($path, $field, $this->tab($label, $settings));
    }

    /**
     * Aggiunge la struttura di base della riga dei Repeater per l'amministrazione
     *
     * @param $path
     */
    public function addRepeaterRow($path)
    {
        $this->addFields(
            $path,
            array(
                'main'			=> array(
                    'class'			=> 'Application\\Admin\\Form\\Repeater\\RowMain',
                ),
                'content'		=> array(
                    'class'			=> 'Application\\Admin\\Form\\Repeater\\RowContent',
                ),
            )
        );
    }
}
