<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Admin
 */

namespace Application\Admin\Dashboard\Widget;

use Pongho\Http\JsonResponse;
use Pongho\Http\Request;
use Pongho\Utilities\DateTime;

/**
 * Widget per Google Analytics.
 */
class VisitsReportWidget extends GoogleAnalyticsWidget
{
	/**
	 * {@inheritdoc}
	 */
	public function getKey()
	{
		return 'visitsreport';
	}

	/**
	 * {@inheritdoc}
	 */
	public function getName()
	{
		return $this->container->getService('translator')->trans('Visitors overview');
	}

	/**
	 * {@inheritdoc}
	 */
	public function getColumns()
	{
		return 2;
	}

	/**
	 * {@inheritdoc}
	 */
	public function render()
	{
		$code = parent::render();

        /** @var \Application\Admin\Controller\DashboardController $controller */
        $controller = $this->container->getService('controller');

        $options = array(
            'url' => $controller->url('/widget/' . $this->getKey() . '/visits/', '&'),
        );

		$head = $this->container->getService('template_head');
		$head->addJavascriptInline('var widget = widget || {}; widget.' . $this->getKey() . ' = ' . json_encode($options) . '; google.load("visualization", "1", {packages: ["corechart"]});');
		$head->addJavascript(pongho_url('/Application/Admin/Resources/public/js/widget/' . $this->getKey() . '.js'));

		return $code . '<div id="visitsreport-chart" class="loading"></div>';
	}

	/**
	 * Risponde alla richiesta /pongho/widget/visitsreport/visits/
	 *
	 * La richiesta non viene fatta direttamente dal JavaScript, perché si richiederebbe anche l’autenticazione OAuth,
	 * che essendo diversa per ogni sito risulta complicata da gestire.
	 *
	 * @access protected
	 * @param \Pongho\Http\Request $request
	 * @return \Pongho\Http\JsonResponse
	 */
	protected function visitsRequestHandler(Request $request)
	{
		$cache = $this->container->getService('cache');

		if ( $cache->exists('gapi-visits-' . $this->getGapiProfile()) )
		{
			$response = $cache->get('gapi-visits-' . $this->getGapiProfile());
		}
		else
		{
			try
			{
				$response = $this->getVisitsStats();
			}
			catch ( \Exception $e )
			{
				return new JsonResponse(array('error' => $e->getMessage()));
			}

			$cache->save('gapi-visits-' . $this->getGapiProfile(), $response, 86400);
		}

		return new JsonResponse($response);
	}

	/**
	 * Restituisce le statistiche sulle visite.
	 *
	 * @access private
	 * @return array
	 */
	private function getVisitsStats()
	{
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->container->getService('translator');

		// Eseguo la richiesta alle API di Google Analytics
		$gapi = $this->getGapiInstance();
		$startDate = new DateTime('-31 day');
		$endDate = new DateTime('-1 day');

		$gapi->requestReportData($this->getGapiProfile(), array('date'), array('visits'), 'date', null, $startDate->format('Y-m-d'), $endDate->format('Y-m-d'), 1, 30);

		// Elaboro il risultato
		$visits = array(
			array($translator->trans('Day'), $translator->trans('Visits'))
		);

		$max = 0;
		foreach ( $gapi->getResults() as $result )
		{
			// Verifico il valore massimo delle visite giornaliere
			if ( $result->getVisits() > $max )
			{
				$max = $result->getVisits();
			}

			// Aggiungo le visite del giorno alla risposta
			$date = new DateTime($result->getDate());

			$visits[] = array(
				$date->localeFormat('%e %B'),
				$result->getVisits()
			);
		}

		$gapi->requestReportData($this->getGapiProfile(), null, array('visits', 'visitors', 'pageviews', 'avgTimeOnSite', 'visitBounceRate', 'percentNewVisits'), array(), null, $startDate->format('Y-m-d'), $endDate->format('Y-m-d'), 1, 30);
		$results = $gapi->getResults();
		$metrics = $results[0]->getMetrics();

		return array(
			'title'		=> sprintf('%s - %s', $startDate->localeFormat('%e %B %Y'), $endDate->localeFormat('%e %B %Y')),
			'max'		=> $max,
			'visits'	=> $visits,
			'metrics'	=> array(
				'visits'			=> array('Visite', number_format($metrics['visits'], 0, ',', '.')),
				'visitors'			=> array('Visitatori unici', number_format($metrics['visitors'], 0, ',', '.')),
				'pageviews'			=> array('Visualizzazioni di pagina', number_format($metrics['pageviews'], 0, ',', '.')),
				'avgTimeOnSite'		=> array('Durata media visita', date('i:s', $metrics['avgTimeOnSite'])),
				'visitBounceRate'	=> array('Frequenza di rimbalzo', round($metrics['visitBounceRate'], 2) . '%'),
				'percentNewVisits'	=> array('% nuove visite', round($metrics['percentNewVisits'], 2) . '%'),
			)
		);
	}
}
