<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Admin
 */

namespace Application\Admin\Dashboard\Widget;

use Pongho\Http\JsonResponse;
use Pongho\Http\Request;
use Pongho\Utilities\DateTime;

class PageviewsReportWidget extends GoogleAnalyticsWidget
{
	/**
	 * {@inheritdoc}
	 */
	public function getKey()
	{
		return 'pageviewsreport';
	}

	/**
	 * {@inheritdoc}
	 */
	public function getName()
	{
		return $this->container->getService('translator')->trans('Most visited pages');
	}

	/**
	 * {@inheritdoc}
	 */
	public function getColumns()
	{
		return 1;
	}

	/**
	 * {@inheritdoc}
	 */
	public function render()
	{
        /** @var \Application\Admin\Controller\DashboardController $controller */
        $controller = $this->container->getService('controller');

        $options = array(
            'url' =>  $controller->url('/widget/' . $this->getKey() . '/stats/', '&'),
        );

		$head = $this->container->getService('template_head');
		$head->addJavascriptInline('var widget = widget || {}; widget.' . $this->getKey() . ' = ' . json_encode($options) . ';');
		$head->addJavascript(pongho_url('/Application/Admin/Resources/public/js/widget/' . $this->getKey() . '.js'));

		return '<div id="pageviewsreport-table" class="loading"></div>';
	}

	/**
	 * Risponde alla richiesta /pongho/widget/pageviewsreport/stats/
	 *
	 * La richiesta non viene fatta direttamente dal JavaScript, perché si richiederebbe anche l’autenticazione OAuth,
	 * che essendo diversa per ogni sito risulta complicata da gestire.
	 *
	 * @access protected
	 * @param \Pongho\Http\Request $request
	 * @return \Pongho\Http\JsonResponse
	 */
	protected function statsRequestHandler(Request $request)
	{
		$cache = $this->container->getService('cache');

		if ( $cache->exists('gapi-pageviews-' . $this->getGapiProfile()) )
		{
			$response = $cache->get('gapi-pageviews-' . $this->getGapiProfile());
		}
		else
		{
			try
			{
				$response = $this->getPageStats();
			}
			catch ( \Exception $e )
			{
				return new JsonResponse(array('error' => $e->getMessage()));
			}

			$cache->save('gapi-pageviews-' . $this->getGapiProfile(), $response, 86400);
		}

		return new JsonResponse($response);
	}

	/**
	 * Restituisce le statistiche sulle pagine più visitate.
	 *
	 * @access private
	 * @return array
	 */
	private function getPageStats()
	{
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->container->getService('translator');

        // Eseguo la richiesta alle API di Google Analytics
		$gapi = $this->getGapiInstance();
		$startDate = new DateTime('-31 day');
		$endDate = new DateTime('-1 day');

		$gapi->requestReportData($this->getGapiProfile(), array('pageTitle', 'pagePath'), array('pageviews'), '-pageviews', null, $startDate->format('Y-m-d'), $endDate->format('Y-m-d'), 1, 10);

		$code = '<table><tr><th>' . $translator->trans('Page') . '</th><th>' . $translator->trans('Visits') . '</th></tr>';

		foreach ( $gapi->getResults() as $result )
		{
			$code .= '<tr><td><span>' . $result->getPageTitle() . '</span><a class="path" href="' . absolute_url($result->getPagePath()) . '">' . absolute_url($result->getPagePath()) . '</a></td><td>' . $result->getPageviews() . '</td></tr>';
		}

		$code .= '</table>';

		return array('code' => $code);
	}
}
