<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Admin
 */

namespace Application\Admin\Dashboard\Widget;

use Pongho\Http\JsonResponse;
use Pongho\Http\Request;
use Pongho\Utilities\DateTime;

class LocationsReportWidget extends GoogleAnalyticsWidget
{
	/**
	 * {@inheritdoc}
	 */
	public function getKey()
	{
		return 'locationsreport';
	}

	/**
	 * {@inheritdoc}
	 */
	public function getName()
	{
		return $this->container->getService('translator')->trans('Visits by country');
	}

	/**
	 * {@inheritdoc}
	 */
	public function getColumns()
	{
		return 1;
	}

	/**
	 * {@inheritdoc}
	 */
	public function render()
	{
		$code = parent::render();

        /** @var \Application\Admin\Controller\DashboardController $controller */
        $controller = $this->container->getService('controller');

        $options = array(
            'url' =>  $controller->url('/widget/' . $this->getKey() . '/locations/', '&'),
        );

		$head = $this->container->getService('template_head');
		$head->addJavascriptInline('var widget = widget || {}; widget.' . $this->getKey() . ' = ' . json_encode($options) . '; google.load("visualization", "1", {packages: ["corechart"]});');
		$head->addJavascript(pongho_url('/Application/Admin/Resources/public/js/widget/' . $this->getKey() . '.js'));

		return $code . '<div id="locationsreport-chart" class="loading"></div>';
	}

	/**
	 * Risponde alla richiesta /pongho/widget/locationsreport/locations/
	 *
	 * La richiesta non viene fatta direttamente dal JavaScript, perché si richiederebbe anche l’autenticazione OAuth,
	 * che essendo diversa per ogni sito risulta complicata da gestire.
	 *
	 * @access protected
	 * @param \Pongho\Http\Request $request
	 * @return \Pongho\Http\JsonResponse
	 */
	protected function locationsRequestHandler(Request $request)
	{
		$cache = $this->container->getService('cache');

		if ( $cache->exists('gapi-locations-' . $this->getGapiProfile()) )
		{
			$response = $cache->get('gapi-locations-' . $this->getGapiProfile());
		}
		else
		{
			try
			{
				$response = $this->getLocationsStats();
			}
			catch ( \Exception $e )
			{
				return new JsonResponse(array('error' => $e->getMessage()));
			}

			$cache->save('gapi-locations-' . $this->getGapiProfile(), $response, 86400);
		}

		return new JsonResponse($response);
	}

	/**
	 * Restituisce le statistiche sui paesi.
	 *
	 * @access private
	 * @return array
	 */
	private function getLocationsStats()
	{
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->container->getService('translator');

		// Eseguo la richiesta alle API di Google Analytics
		$gapi = $this->getGapiInstance();
		$startDate = new DateTime('-31 day');
		$endDate = new DateTime('-1 day');

		$gapi->requestReportData($this->getGapiProfile(), null, array('visits'), array(), null, $startDate->format('Y-m-d'), $endDate->format('Y-m-d'), 1, 1);
		$results = $gapi->getResults();
		$total_visits = $results[0]->getVisits();

		$gapi->requestReportData($this->getGapiProfile(), array('country'), array('visits'), '-visits', null, $startDate->format('Y-m-d'), $endDate->format('Y-m-d'), 1, 6);

		$stats = array(
			array($translator->trans('Country'), $translator->trans('Visits')),
		);
		$other = $total_visits;
		foreach ( $gapi->getResults() as $result )
		{
			$stats[] = array($result->getCountry(), $result->getVisits());
			$other -= $result->getVisits();
		}

		$stats[] = array($translator->trans('Other countries'), $other);

		return array(
			'title'			=> $this->getName(),
			'totalvisits'	=> $total_visits,
			'stats'			=> $stats,
		);
	}
}
