<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Admin
 */

namespace Application\Admin\Dashboard\Widget;

use Pongho\Http\JsonResponse;
use Pongho\Http\Request;
use Pongho\Utilities\DateTime;

class KeywordsReportWidget extends GoogleAnalyticsWidget
{
	/**
	 * {@inheritdoc}
	 */
	public function getKey()
	{
		return 'keywordsreport';
	}

	/**
	 * {@inheritdoc}
	 */
	public function getName()
	{
		return $this->container->getService('translator')->trans('Organic search traffic');
	}

	/**
	 * {@inheritdoc}
	 */
	public function getColumns()
	{
		return 1;
	}

	/**
	 * {@inheritdoc}
	 */
	public function render()
	{
        /** @var \Application\Admin\Controller\DashboardController $controller */
        $controller = $this->container->getService('controller');

        $options = array(
            'url' =>  $controller->url('/widget/' . $this->getKey() . '/keywords/', '&'),
        );

		$head = $this->container->getService('template_head');
		$head->addJavascriptInline('var widget = widget || {}; widget.' . $this->getKey() . ' = ' . json_encode($options) . ';');
		$head->addJavascript(pongho_url('/Application/Admin/Resources/public/js/widget/' . $this->getKey() . '.js'));

		return '<div id="keywordsreport-table" class="loading"></div>';
	}

	/**
	 * Risponde alla richiesta /pongho/widget/keywordsreport/keywords/
	 *
	 * La richiesta non viene fatta direttamente dal JavaScript, perché si richiederebbe anche l’autenticazione OAuth,
	 * che essendo diversa per ogni sito risulta complicata da gestire.
	 *
	 * @access protected
	 * @param \Pongho\Http\Request $request
	 * @return \Pongho\Http\JsonResponse
	 */
	protected function keywordsRequestHandler(Request $request)
	{
		$cache = $this->container->getService('cache');

		if ( $cache->exists('gapi-keywords-' . $this->getGapiProfile()) )
		{
			$response = $cache->get('gapi-keywords-' . $this->getGapiProfile());
		}
		else
		{
			try
			{
				$response = $this->getKeywordsStats();
			}
			catch ( \Exception $e )
			{
				return new JsonResponse(array('error' => $e->getMessage()));
			}

			$cache->save('gapi-keywords-' . $this->getGapiProfile(), $response, 86400);
		}

		return new JsonResponse($response);
	}

	/**
	 * Restituisce le statistiche sulle parole chiave.
	 *
	 * @access private
	 * @return array
	 */
	private function getKeywordsStats()
	{
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->container->getService('translator');

		// Eseguo la richiesta alle API di Google Analytics
		$gapi = $this->getGapiInstance();
		$startDate = new DateTime('-31 day');
		$endDate = new DateTime('-1 day');

		$gapi->requestReportData($this->getGapiProfile(), array('keyword'), array('visits'), '-visits', null, $startDate->format('Y-m-d'), $endDate->format('Y-m-d'), 1, 11);

		$code = '<table><tr><th>' . $translator->trans('Keyword') . '</th><th>' . $translator->trans('Visits') . '</th></tr>';

		foreach ( $gapi->getResults() as $result )
		{
			if ( $result->getKeyword() === '(not set)' )
			{
				continue;
			}

			$code .= '<tr><td>' . $result->getKeyword() . '</td><td>' . $result->getVisits() . '</td></tr>';
		}

		$code .= '</table>';

		return array('code' => $code);
	}
}
