<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Admin
 */

namespace Application\Admin\Controller;

use Application\Admin\Kernel;
use Pongho\Form\Form;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Response;

/**
 * Class CrudFormController
 */
abstract class CrudFormController extends Crud2Controller
{
	/**
	 * @param \Pongho\Form\Form $form
	 * @return string
	 */
    protected function renderForm(Form $form)
    {
        return $form->render();
    }

	/**
	 * @param \Pongho\Form\Form $form
	 * @return null
	 */
    protected function renderAddEdit(Form $form)
    {
        $this->getHelper()->getHead()
            ->addJavaScript(pongho_url('/vendor/jQueryUI-widgets/nestedSortable/jquery.ui.nestedSortable.js?v=' . filemtime(PONGHO_PATH . '/vendor/jQueryUI-widgets/nestedSortable/jquery.ui.nestedSortable.js')))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/views/js/plugins.js?v=' . Kernel::VERSION))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/views/js/scripts.js?v=' . Kernel::VERSION));

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../Resources/views/edit.php')
            ->assignVars(array(
                'title' 	=> $this->getAddEditTitle($form->getSubject()->getModel()),
                'content'	=> $form,
            ));

        return null;
    }

    /**
     * @return \Application\Admin\Form\FormConfig
     */
    abstract protected function getFormsOptions();

    /**
     * @return null|RedirectResponse|Response
     */
    protected function addEdit()
    {
        $path = $this->getParameter('path');
        $form = $this->getForm();
        $model = $form->getSubject()->getModel();

        $this->getHelper()->notify($this, $this->getEventNamespace('notify_before_render_view'), array('model' => $model));

        $this->invokeCallBack('before_set_template', $model);

        $this->invokeCallBack('after_initialize', $form);

        if ($this->getRequest()->getMethod() === 'POST') {
            $this->getHelper()->notify($this, $this->getEventNamespace('before_save'), array('form' => $form, 'model' => $model));
            $this->invokeCallBack('before_save', $form);

            $form->handleRequest($this->getRequest());

            if (!$form->hasErrors()) {
                $this->invokeCallBack('after_save', $form, array());
                $this->getHelper()->notify($this, $this->getEventNamespace('after_save'), array('form' => $form, 'model' => $model));
            }

            if ($this->response instanceof Response) {
                return $this->response;
            }

            if ($this->action === 'add') {
                if (!$form->hasErrors()) {
                    return $this->getHelper()->redirectResponse($this->url('/' . $path . '/edit/' . $model->id . '/', '&'));
                }
            } else {
                return new Response($this->renderForm($form));
            }
        }

        $this->renderAddEdit($form);

        $this->getHelper()->notify($this, $this->getEventNamespace('notify_after_render_view'), array('model' => $model));

        return null;
    }

    /**
	 * @return \Application\Admin\Form\FormBuilder
	 */
	protected function getFormBuilderClass()
	{
		return 'Application\\Admin\\Form\\FormBuilder';
	}

	/**
	 * @return \Application\Admin\Form\Generator\AdminGenerator
	 */
	protected function getFormGeneratorClass()
	{
		return 'Application\\Admin\\Form\\Generator\\AdminGenerator';
	}

	/**
	 * @return string
	 */
	protected function getFormAction()
	{
		$path = $this->getParameter('path');

		// Gestione dell'azione
		if ( $this->action === 'add' )
		{
			return $this->url('/' . $path . '/add/');
		}
		else
		{
			return $this->url('/' . $path . '/edit/' . $this->getModel()->id . '/');
		}
	}

	/**
	 * @return \Pongho\Form\Form
	 */
	protected function getForm()
	{
		/** @var $form \Pongho\Form\Form */
		static $form;

		if ( $form === null )
		{
			$builder = $this->getFormBuilderClass();
			$generator = $this->getFormGeneratorClass();

			$form = $builder::buildConfig($this->getFormsOptions());

			$form->setGenerator(new $generator($this->getHelper()->getLocalization()));
			$form->setAction($this->getFormAction());
		}

		return $form;
	}

    /**
     * Restituisce la chiave della tab che deve essere attivata
     *
     * @return string
     */
    protected function getActiveTab()
    {
        $tab = $this->getRequest()->post->get('tabs[tab]', 'main', true);
        $tab = $this->getRequest()->query->get('tabs[tab]', $tab, true);

        return $tab;
    }


    /**
     * callback after_initialize
     *
     * Serve a modificare il formato dei dati nella Request, i menu più grandi potrebbero arrivare al limite di campi HTTP
     * serializzando i campi ed inviando il dato serializzato si risolve il problema
     *
     * I campi della Form mi arrivano serializzati, devo deserializzarli prima di farli gestire alla Form
     *
     * @param Form $form
     */
    public function afterInitializeCallback(Form $form)
    {
        $form->setAttribute('data-request-format', 'serialized');

        /** @var \Pongho\Http\Request $request */
        $request = $this->getRequest();

        $post = $request->post->get('form');

        // Se è arrivata la form serializzata riscrivo la Request in modo da passare alla Form quanto si aspetta.
        if ($post) {
            $post = $this->deepConvertStdClassToArray(json_decode($post));

            foreach ($post as $name => $value) {
                $request->post->set($name, $value);
            }
        }
    }

    /**
     * deepConvertStdClassToArray
     *
     * @param array|\stdClass $value
     * @return array
     */
    private function deepConvertStdClassToArray($value)
    {
        if ($value instanceof \stdClass) {
            $value = (array) $value;
        }

        if (is_array($value)) {
            $new_value = array();

            foreach ($value as $key => &$field_value) {
                if (is_numeric($key)) {
                    $key = (int) $key;
                }

                $new_value[$key] = $this->deepConvertStdClassToArray($field_value);
            }

            $value = $new_value;
        }

        return $value;
    }
}
