<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Admin
 */

namespace Application\Admin\Controller;

use ActiveRecord\Column;
use Application\Admin\Form\FilterSubject;
use Application\Admin\Form\FormBuilder;
use Application\Admin\Form\FormConfig;
use Application\Admin\Form\Generator\AdminGenerator;
use Application\Admin\Model\Filter;
use Application\Admin\Utilities\ArchiveManager;
use Application\Cms\Model\CustomFieldValue;
use Application\Core\Model\Account;
use Application\Core\Model\File;
use Application\Core\Utilities\AssociativeArrayObject;
use Application\Core\Utilities\Pagination;
use Pongho\Form\Fieldset;
use Pongho\Form\Form;
use Pongho\Form\Repeater\RepeaterRow;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpException;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\JsonResponse;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Response;
use Pongho\Template\View;
use Pongho\Template\ViewClosure;
use Pongho\Utilities\Inflector;

/**
 * Class Crud2Controller
 */
abstract class Crud2Controller extends AdminController
{
    /** @var \Application\Admin\Utilities\ArchiveManager */
    protected $am;

    protected $page = 1;

    /**
     * Restituisce il namespace per il salvataggio dei filtri
     *
     * @return string
     */
    abstract function getNamespace();

    /**
     * Restituisce il namespace usato negli eventi
     *
     * @param $event_name
     * @return string
     */
    abstract function getEventNamespace($event_name);

    /**
     * Restituisce le opzioni per il modello dell'elenco
     *
     * @return array
     */
    protected function getModelOptions()
    {
        return array();
    }

    /**
     * Restituisce i campi del modello filtrabili tramite ricerca libera
     *
     * @return array
     */
    protected function getSearchFields()
    {
        return array();
    }

    /**
     * Restituisce la configurazione dei campi filtrabili tramite i filtri
     *
     * @return array
     */
    protected function getModelFieldsConfig()
    {
        return array();
    }

    /**
     * Restituisce la stringa del titolo dell'archivio
     *
     * @return string
     */
    abstract protected function getArchiveTitle();

    /**
     * Restituisce il testo del pulsante "add_new"
     *
     * @return string
     */
    abstract protected function getArchiveAddButtonText();

    /**
     * Restituisce il testo per il link "add_first"
     *
     * @return string
     */
    abstract protected function getArchiveAddFirstText();

    /**
     * Restituisce il testo per la tabella quando non contiene record e non sono in ricerca
     *
     * @return string
     */
    protected function getArchiveEmptyRecordsetText()
    {
        return $this->getContainer()->getService('translator')->trans('No records to show yet');
    }

    /**
     * Indica se si ha il permesso di utilizzare l'azione "add"
     *
     * @return bool
     */
    protected function canAdd()
    {
        return false;
    }

    /**
     * Indica al template se visualizzare il tasto "Aggiungi"
     *
     * @return bool
     */
    protected function hasAddButton()
    {
        return $this->canAdd();
    }

    /**
     * Restituisce il testo per il link "search_no_results"
     *
     * @return string
     */
    protected function getArchiveSearchNoResultsText()
    {
        return $this->getContainer()->getService('translator')->trans('The search did not return any result');
    }

    /**
     * Restituisce i valori per i campi enum dei filtri
     *
     * @return array
     *
     * @example
     * $fields = array(
     *   'fieldname' => array(
     *     'value_1' => 'label_1',
     *     'value_2' => 'label_2',
     *   ),
     * );
     */
    protected function getFilterEnumValues()
    {
        return array();
    }

    /**
     * Restituisce i valori per i campi select dei filtri (un array di funzioni anonime)
     *
     * @return array
     *
     * @example
     * $fields = array(
     *   'fieldname' => function() {
     *     $values = array();
     *     foreach ( Model::all() as $model )
     *     {
     *       $values[$model->id] = $model->label;
     *     }
     *     return $values;
     *   },
     * );
     */
    protected function getFilterSelectValues()
    {
        return array();
    }

    /**
     * Restituisce i valori per i filtri rapidi
     *
     * @return array
     */
    protected function getFastFilters()
    {
        return array();
    }

    /**
     * Restituisce il nome della classe del modello.
     *
     * @return string
     */
    abstract public function getModelClass();

    /**
     * Restituisce la lista dei campi da elaborare.
     *
     * @param mixed $model
     * @return string
     */
    abstract public function getAddEditTitle($model);

    /**
     * @return null|RedirectResponse|Response
     */
    abstract protected function addEdit();

    /**
     * Controlla quali CustomField applicare al modello corrente e li aggiunge alla Form
     *
     * @param \Application\Admin\Form\FormConfig $config
     * @return \Application\Admin\Form\FormConfig
     */
    protected function addCustomFieldsToFormFields(FormConfig $config)
    {
        return $config;
    }

    /**
     * Restituisce le opzioni di compatibilità delle regole
     *
     * @param \Pongho\Form\Subject\ModelSubject $subject
     * @return array
     */
    protected function getCustomFieldRulesOptions(ModelSubject $subject)
    {
        return array();
    }

    /**
     * Azione per eliminare un file caricato tramite custom fields
     *
     * @access public
     */
    public function deletecustomfileAction()
    {
        $custom_field_id = $this->getRequest()->query->get('field');

        /** @var CustomFieldValue $custom_field */
        if (($custom_field = CustomFieldValue::find($custom_field_id)) !== null) {
            if (($file = File::find($custom_field->field_value)) !== null) {
                if ($file->delete()) {
                    $custom_field->delete();
                }
            }
        }

        if ($this->getRequest()->isAjax()) {
            return $this->getHelper()->displayJsonMessage($this->getContainer()->getService('translator')->trans('The file has been deleted'));
        }

        return new RedirectResponse($this->url(sprintf('/%s/edit/%d/', $this->getParameter('path'), $this->getParameter('id'))));
    }

    /**
     * @return array
     */
    protected function getTableColumns()
    {
        return array();
    }

    /**
     * Restituisce il percorso del template per l'archivio del modello
     *
     * @return string
     */
    public function getArchiveTemplatePath()
    {
        return __DIR__ . '/../Resources/views/list.php';
    }

    /**
     * Restituisce il percorso del template per la tabella dell'archivio
     *
     * @return string
     */
    protected function getArchiveTableTemplatePath()
    {
        return __DIR__ . '/../Resources/views/list_table.php';
    }

    /**
     * Azione `index`.
     *
     * @return bool|null|Response
     */
    public function indexAction()
    {
        $response = $this->getHelper()->notifyUntil($this, $this->getEventNamespace('notify_index'));

        if ($response instanceof Response) {
            return $response;
        }

        return $this->displayArchive();
    }

    /**
     * Azione `page`.
     *
     * @return null|Response
     */
    public function pageAction()
    {
        $this->page = $this->getParameter('page', 1, true);

        return $this->displayArchive();
    }

    /**
     * Visualizza l’archivio dei record.
     *
     * @return null|\Pongho\Http\Response
     */
    public function displayArchive()
    {
        $filter_id = intval($this->getRequest()->query->get('filter'));

        $am = $this->getArchiveManager();

        $this->getHelper()->notify($this, $this->getEventNamespace('notify_archive_before_filter'), array('archive_manager' => $am));

        // Provo a caricare un filtro, se non ho specificato un id oppure non trovo il filtro restituisco un nuovo filtro
        /** @var Filter $filter */
        if (!$filter_id || ($filter = Filter::find($filter_id)) === null) {
            $f = $this->getRequest()->query->get('f', array());
            $am->setFilters($f);

            $filter = new Filter();
// todo: non funziona senza impostare correttamente i campi utilizzati dal filtro, ma è difficile ricalcolarli partendo da GET
// todo: quindi se si apre un link con un filtro in una nuova scheda, al momento la form del filtro sarà vuota
//            $f = array();
//
//            foreach ($filters as $field => $filter) {
//                foreach ($filter as $operator => $value)
//                    $f[] = array(
//                        'field' => $field,
//                        'operator' => $operator,
//                        'type'  => 'string',
//                        'valuefield' => 'value_text',
//                        'value_text' => $value,
//                    );
//            }
//
//            $filter->filter = serialize($f);
        } else {
            $am->setFilters($filter->getFilters());
        }

        $form = $am->getFiltersForm($filter);

        // Se sono in POST, salvo il filtro
        if ($this->getRequest()->getMethod() === 'POST') {
            // Controllo che ci sia tutto quello che mi serve per salvare il filtro
            $filters = $this->getRequest()->post->get('filters', array());

            if ($filters['name'] === '') {
                $form->addError('name', $this->getContainer()->getService('translator')->get('To save a filter the name is required'));
            }

            $form
                ->setValue('namespace', $this->getNamespace())
                ->setValue('user_id', $this->getContainer()->getService('current_user')->id)
                ->handleRequest($this->getRequest());

            // Ricarico la form dei filtri e l'elenco dei filtri
            return new JsonResponse(array(
                'filters' => $this->getFiltersListHtml(),
                'form'    => $form->render(),
            ));
        }

        $this->getHelper()->notify($this, $this->getEventNamespace('notify_archive_after_filter'), array('archive_manager' => $am));

        // Se sono in AJAX restituisco solo le parti che cambiano
        if ($this->getRequest()->isAjax()) {
            $list = new View($this->getArchiveTableTemplatePath());

            $list
                ->assignVars($this->getHelper()->getViewGlobalVariables())
                ->assignVars(
                    array(
                        'mass_action_url'     => $this->getMassActionUrl(),
                        'mass_action_buttons' => $this->getMassActionButtons(),
                        'pagination'          => $this->getArchiveManager()->getPagination(),
                        'table'               => $this->getArchiveManager()->getTable(),
                    )
                );

            $this->beforeRenderArchive();

            return new JsonResponse(array(
                'list'         => $list->render(),
                'form'         => $form->render(),
                'fast_filters' => $this->getArchiveManager()->isFastFiltersEnabled() ? $this->getArchiveManager()->getFastFilters()->render() : '',
            ));
        }

        // In tutti gli altri casi...
        $operators = json_encode($this->getHelper()->filter($this, $this->getEventNamespace('filter_filters_operators'), $this->getArchiveManager()->getOperators()));
        $enums = json_encode($this->getFilterEnumValuesWrapper());

        $this->getHelper()->getHead()
            ->addJavaScript(pongho_url('/vendor/jQueryUI-widgets/nestedSortable/jquery.ui.nestedSortable.js'))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/views/js/plugins.js'))
            ->addJavaScript(pongho_url('/vendor/jQuery-plugins/URLParser/purl.js'))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/views/js/jquery.filters.js'))
            ->addJavaScriptInline("$('#filters').data('operators', {$operators}).data('enums', {$enums});");

        $this->getHelper()->getBodyView()
            ->setTemplatePath($this->getArchiveTemplatePath())
            ->assignVars(
                array(
                    'title'             => $this->getArchiveTitle(),
                    'left_header_view'  => $this->getLeftHeaderCode(),
                    'right_header_view' => $this->getRightHeaderCode(),
                    'filters_view'      => $this->getFiltersCode($form),
                    'table_view'        => $this->getTableCode(),
                )
            );

        $this->beforeRenderArchive();

        $this->getHelper()->notify($this, $this->getEventNamespace('notify_before_render_archive'));

        return null;
    }

    /**
     * Controlla il permesso per utilizzare l'azione
     * @throws HttpUnauthorizedException
     */
    protected function checkAddPermit()
    {
    }

    /**
     * Controlla il permesso per utilizzare l'azione
     * @throws HttpUnauthorizedException
     */
    protected function checkEditPermit()
    {
    }

    /**
     * Controlla il permesso per utilizzare l'azione
     * @throws HttpUnauthorizedException
     */
    protected function checkDeletePermit()
    {
    }

    /**
     * Controlla il permesso per utilizzare l'azione
     * @throws HttpUnauthorizedException
     */
    protected function checkClonePermit()
    {
    }

    /**
     * Controlla il permesso per utilizzare l'azione
     * @throws HttpUnauthorizedException
     */
    protected function checkMassPermit()
    {
    }

    /**
     * Azione `add`.
     *
     * @return bool|null|RedirectResponse|Response
     */
    public function addAction()
    {
        $this->checkAddPermit();

        $response = $this->getHelper()->notifyUntil($this, $this->getEventNamespace('notify_add'));

        if ($response instanceof Response) {
            return $response;
        }

        return $this->addEdit();
    }

    /**
     * Azione `edit`.
     *
     * @return bool|null|RedirectResponse|Response
     */
    public function editAction()
    {
        $this->checkEditPermit();

        $response = $this->getHelper()->notifyUntil($this, $this->getEventNamespace('notify_edit'));

        if ($response instanceof Response) {
            return $response;
        }

        return $this->addEdit();
    }

    /**
     * Azione `clone`.
     *
     * @return null|RedirectResponse|Response
     * @ignore
     */
    public function cloneAction()
    {
        $this->checkClonePermit();

        return $this->addEdit();
    }

    /**
     * Azione `delete`.
     *
     * @return JsonResponse|RedirectResponse
     * @throws \Exception
     */
    public function deleteAction()
    {
        $this->checkDeletePermit();

        if (($model = $this->getModel()) === null) {
            throw new HttpNotFoundException('Record not found!');
        }

        try {
            $this->invokeCallBack('before_delete');

            $delete = $model->transaction(
                function () use ($model) {
                    return $model->delete();
                }
            );

            if ($delete) {
                $this->invokeCallBack('after_delete');

                if ($this->getRequest()->isAjax()) {
                    return $this->getHelper()->displayJsonMessage('Riga eliminata!');
                } else {
                    return new RedirectResponse($this->url('/' . $this->getParameter('path') . '/'));
                }
            }

            throw new \RuntimeException('Could not delete the record!');
        } catch (\Exception $e) {
            if ($this->getRequest()->isAjax()) {
                return $this->getHelper()->displayJsonError($e->getMessage());
            } else {
                throw $e;
            }
        }
    }

    /**
     * Azione `mass`.
     *
     * @return RedirectResponse
     * @throws \ActiveRecord\Exceptions\ActiveRecordException
     * @throws \Exception
     */
    public function massAction()
    {
        if ($this->getRequest()->getMethod() !== 'POST') {
            throw new HttpException(405);
        }

        $this->checkMassPermit();

        $ids = isset($_POST['ids']) ? $_POST['ids'] : array();
        $action = isset($_POST['mass_action']) ? $_POST['mass_action'] : null;

        if (!empty($ids)) {
            /** @var \ActiveRecord\Base $model_class */
            $model_class = $this->getModelClass();

            if (method_exists($model_class, $action)) {
                // Le azioni di massa devono essere eseguite caricando un modello alla volta e non tutti assieme,
                // perché quando un modello viene modificato potrebbe modificare anche gli altri senza che questi
                // vengano aggiornati. È il caso, ad esempio, delle azioni "Sposta nel cestino" e "Archivia" del CMS,
                // dove le posizioni left e right dei nodi vengono spostate.
                foreach (array_keys($ids) as $id) {
                    if (($model = $model_class::find($id)) === null) {
                        continue;
                    }

                    $model->transaction(
                        function () use ($model, $action) {
                            return $model->$action();
                        }
                    );
                }
            }
        }

        return new RedirectResponse($this->getMassActionRedirectUrl());
    }

    /**
     * Restituisce il percorso del template da utilizzare per le action add e edit.
     *
     * @throws \UnexpectedValueException
     * @return string
     *
     * @todo   Quando passeremo alla gestione delle form di amministrazione con le
     *       Pongho\Form, potremo usare un template generico per le pagine add/edit.
     */
    protected function getAddEditTemplatePath()
    {
        $ref = new \ReflectionClass($this);
        $class_dir = dirname($ref->getFileName());
        $views_dir = $class_dir . '/../../Resources/views';

        if (!file_exists($views_dir) || !is_dir($views_dir)) {
            throw new \UnexpectedValueException(sprintf('Directory "%s" not exists!', $views_dir));
        }

        $class_name = Inflector::demodulize($ref->getName());
        $filepath = $views_dir . '/' . Inflector::underscore(substr($class_name, 0, -10)) . '_edit.php';

        if (!file_exists($filepath)) {
            throw new \UnexpectedValueException(sprintf('File "%s" not exists!', $filepath));
        }

        return $filepath;
    }

    /**
     * Richiama una callback.
     *
     * @throws \InvalidArgumentException
     * @return void
     */
    protected function invokeCallBack( /* $callback_name, $args */)
    {
        $args = func_get_args();

        if (!isset($args[0])) {
            throw new \InvalidArgumentException(sprintf('Method "%s()" expects at least 1 argument, 0 given!', __METHOD__));
        }

        $callback_method = Inflector::camelize(array_shift($args), false) . 'Callback';

        if (method_exists($this, $callback_method)) {
            call_user_func_array(array($this, $callback_method), $args);
        }
    }

    /**
     * Salva il modello.
     *
     * @param mixed $model
     * @return boolean
     */
    protected function saveModel($model)
    {
        return $model->save();
    }

    /**
     * Indica se è stato eseguito l’invio del modulo.
     *
     * @return bool
     */
    protected function isSubmit()
    {
        return $this->getRequest()->getMethod() === 'POST';
    }

    /**
     * Restituisce il modello in base all’azione.
     *
     * @return mixed
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    protected function getModel()
    {
        /** @var \ActiveRecord\Base $model_class */
        $model_class = $this->getModelClass();

        if ($this->action == 'add') {
            return new $model_class();
        } else {
            if (($model = $model_class::find($this->getParameter('id', null, true), array('cachable' => false))) === null) {
                throw new HttpNotFoundException();
            }

            if ($this->action == 'duplicate') {
                return clone $model;
            } else {
                return $model;
            }
        }
    }

    /**
     * Restituisce la risposta dopo il salvatggio del modello.
     *
     * @param $model
     * @return RedirectResponse
     */
    protected function getAddEditResponse($model)
    {
        return new RedirectResponse($this->url(sprintf('/%s/edit/%d/', $this->getParameter('path'), $model->id)));
    }

    /**
     * Restituisce la action per le azioni di massa.
     *
     * @return string
     */
    protected function getMassActionUrl()
    {
        return $this->url('/' . $this->getParameter('path') . '/mass/');
    }

    /**
     * Restituisce il link di redirect per le azioni di massa.
     *
     * @return string
     */
    protected function getMassActionRedirectUrl()
    {
        return $this->url('/' . $this->getParameter('path') . '/');
    }

    /**
     * Restituisce l’elenco delle azioni di massa.
     *
     * @return array
     */
    protected function getMassActions()
    {
        return array('delete' => $this->getContainer()->getService('translator')->trans('Delete'));
    }

    /**
     * Restituisce il codice per visualizzare le azioni di massa.
     *
     * @return string
     */
    protected function getMassActionButtons()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $mass_actions = $this->getMassActions();

        switch (count($mass_actions)) {
            case 0:
                return '';

            case 1:
                $label = reset($mass_actions);
                $action = key($mass_actions);

                return '<input type="hidden" name="mass_action" value="' . $action . '" class="hide" /><input type="submit" class="pongho-button" name="submit" value="' . $label . '" />';

            default:
                $options = '<option>' . $translator->trans('Mass actions') . '</option>';

                foreach ($mass_actions as $action => $label) {
                    $options .= '<option value="' . $action . '">' . $label . '</option>';
                }

                return '<select name="mass_action">' . $options . '</select> <input type="submit" class="pongho-button" value="' . $translator->trans('Submit') . '" name="submit" />';
        }
    }

    /**
     * Visualizza la paginazione
     *
     * @param integer $page          Il numero della pagina corrente.
     * @param integer $total_records Numero totale dei records.
     * @param integer $rows_per_page Numero di righe per pagina da visualizzare.
     * @param array   $query         Elenco dei parametri da aggiungere in query string nei collegamenti.
     */
    protected function displayPagination($page, $total_records, $rows_per_page = null, array $query = array())
    {
        if ($rows_per_page === null) {
            $rows_per_page = $this->records_per_page;
        }

        $url = preg_replace('#(page\/([0-9]+)\/)$#', '', $this->getRequest()->getPathInfo());

        $this->getHelper()->getBodyView()
            ->assignVar('pagination', new Pagination($url, $total_records, $page, $rows_per_page, $query));
    }

    /**
     * Genera la lista delle azioni. L’array da passare al metodo deve essere di questo tipo:
     * <code>
     * $options = array(
     *     array('Edit', 'href' => $this->_url(""), 'title' => 'Edit_message', 'class' => 'edit'),
     *     array('Delete', 'href' => $this->_url(""), 'class' => 'delete'),
     * );
     * </code>
     *
     * @param array $actions
     * @return string
     */
    protected function parseActions(array $actions)
    {
        $html = $this->getContainer()->getService('template_html');

        $code = array();
        foreach ($actions as $action) {
            $label = array_shift($action);
            $attributes = $action;

            $code[] = $html->generateTagBlock('a', $label, $attributes);
        }

        return implode(' | ', $code);
    }

    /**
     * Abilita l'editor wysiwyg (TinyMCE).
     *
     * @param array $options
     * @todo: dipendenza sul Cms (node_type), potrei avere un editor in un'altra applicazione senza avere il Cms installato
     */
    protected function enableWysiwyg(array $options = array())
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $language = $this->getHelper()->getUser()->getAccount()->language->culture;

        if (!file_exists(PONGHO_PATH . '/vendor/TinyMCE/langs/' . $language . '.js')) {
            if (strlen($language) > 2) {
                $language = substr($language, 0, 2);

                if (!file_exists(PONGHO_PATH . '/vendor/TinyMCE/langs/' . $language . '.js')) {
                    $language = 'it';
                }
            } else {
                $language = 'it';
            }
        }

        $options = array_merge(
            array(
                'language'                => $language,
                // Dimensioni
                'width'                   => '100%',
                // HTML5
                'element_format'          => 'html', // sostituisce '<br />' con '<br>'
                'fix_list_elements'       => true, // corregge le liste indentate male
                'schema'                  => 'html5-strict',
                'extended_valid_elements' => 'span[style|id|nam|class|lang]',
                // CSS
                'content_css'             => pongho_url(
                    '/Application/Admin/Resources/views/css/editor-content.css?v=' . filemtime(
                        __DIR__ . '/../Resources/views/css/editor-content.css'
                    )
                ),
                // Caratteri accentati
                'entity_encoding'         => 'raw',
                'convert_urls'            => false,
            ),
            $options
        );

        $options = $this->getHelper()->filter($this, 'admin.filter_wysiwyg_options', $options);
        if (($module = $this->getHelper()->getModuleSite(false)) !== null && $module->node_type) {
            $options = $this->getHelper()->filter($this, $this->getEventNamespace('filter_wysiwyg_options'), $options);
        }

        // Opzioni Normal
        $options_normal = array_merge(
            $options,
            array(
                // Plugins
                'plugins'                   => 'advlist anchor charmap code contextmenu directionality fullscreen hr image link lists media paste print responsivefilemanager searchreplace tabfocus table textcolor visualblocks wordcount',

                // Menu
                'menu'                      => array(
                    'edit'   => array(
                        'title' => $translator->trans('Edit', array(), 'wysiwyg'),
                        'items' => 'undo redo | searchreplace | selectall',
                    ),
                    'format' => array(
                        'title' => $translator->trans('Format', array(), 'wysiwyg'),
                        'items' => 'bold italic underline strikethrough | subscript superscript | formats | removeformat',
                    ),
                    'insert' => array(
                        'title' => $translator->trans('Insert', array(), 'wysiwyg'),
                        'items' => 'link anchor | hr | image media | charmap',
                    ),
                    'table'  => array(
                        'title' => $translator->trans('Table', array(), 'wysiwyg'),
                        'items' => 'inserttable tableprops deletetable cell row column',
                    ),
                    'tools'  => array(
                        'title' => $translator->trans('Tools', array(), 'wysiwyg'),
                        'items' => 'visualblocks fullscreen | print code',
                    ),
                ),

                // Strumenti
                'toolbar'                   => array(
                    'formatselect | bold italic underline forecolor | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | link unlink image media',
                ),

                // Tabelle
                'visual'                    => true,

                // FileManager
                'external_filemanager_path' => pongho_url('/vendor/responsivefilemanager/'),
                'filemanager_title'         => $translator->trans('File manager', array(), 'wysiwyg'),
                'external_plugins'          => array(
                    "filemanager" => pongho_url('/vendor/responsivefilemanager/plugin.min.js'),
                ),

                // Allineamento
                'formats'                   => array(
                    'alignleft'    => array(
                        'selector' => 'p,h1,h2,h3,h4,h5,h6,td,th,div,ul,ol,li,table,img',
                        'classes'  => 'alignleft'
                    ),
                    'alignright'   => array(
                        'selector' => 'p,h1,h2,h3,h4,h5,h6,td,th,div,ul,ol,li,table,img',
                        'classes'  => 'alignright'
                    ),
                    'alignjustify' => array(
                        'selector' => 'p,h1,h2,h3,h4,h5,h6,td,th,div,ul,ol,li,table,img',
                        'classes'  => 'alignfull'
                    ),
                    'aligncenter'  => array(
                        'selector' => 'p,h1,h2,h3,h4,h5,h6,td,th,div,ul,ol,li,table,img',
                        'classes'  => 'aligncenter'
                    )
                ),
            )
        );

        // Opzioni Small
        $options_small = array_merge(
            $options,
            array(
                // Plugins
                'plugins' => 'link wordcount code',

                // Menu
                'menubar' => false,

                // Strumenti
                'toolbar' => array(
                    'bold italic underline bullist numlist link unlink | code',
                ),
            )
        );

        // Opzioni Full
        $options_full = array_merge(
            $options_normal,
            array(
                'plugins' => 'advlist anchor charmap code contextmenu directionality fullpage fullscreen hr image link lists media paste print responsivefilemanager searchreplace tabfocus table textcolor visualblocks wordcount',

                // Strumenti
                'toolbar' => array(
                    'formatselect | fontselect | fontsizeselect | forecolor backcolor | removeformat',
                    'bold italic underline | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | link unlink | image media',
                ),
            )
        );

        // Opzioni Newsletter
        $options_newsletter = array_merge(
            $options_full,
            array(
                // Schema
                'schema'                       => 'html4',

                // Fullpage
                'fullpage_default_doctype'     => '',
                'fullpage_default_encoding'    => 'UTF-8',
                'fullpage_default_title'       => 'Newsletter',
                'fullpage_default_font_size'   => '12px',
                'fullpage_default_font_family' => "'Lucida Grande', Lucida, Verdana, sans-serif",
                'fullpage_default_text_color'  => '#333',
                'fullpage_default_xml_pi'      => false,
                'fullpage_hide_in_source_view' => false,
            )
        );

        unset($options_newsletter['formats']);

        // Filtri
        $options_normal = $this->getHelper()->filter($this, 'admin.filter_wysiwyg_options.normal', $options_normal);
        $options_small = $this->getHelper()->filter($this, 'admin.filter_wysiwyg_options.small', $options_small);
        $options_full = $this->getHelper()->filter($this, 'admin.filter_wysiwyg_options.full', $options_full);
        $options_newsletter = $this->getHelper()->filter($this, 'admin.filter_wysiwyg_options.newsletter', $options_newsletter);

        if (($module = $this->getHelper()->getModuleSite(false)) !== null && $module->node_type) {
            $options_normal = $this->getHelper()->filter($this, $this->getEventNamespace('filter_wysiwyg_options.normal'), $options_normal);
            $options_small = $this->getHelper()->filter($this, $this->getEventNamespace('filter_wysiwyg_options.small'), $options_small);
            $options_full = $this->getHelper()->filter($this, $this->getEventNamespace('filter_wysiwyg_options.full'), $options_full);
        }

        // Javascript
        $options = json_encode(
            array(
                'normal'     => $options_normal,
                'small'      => $options_small,
                'full'       => $options_full,
                'newsletter' => $options_newsletter,
            )
        );

        $host_url = $this->getHelper()->getSite()->domain();

        $this->getHelper()->getHead()
            ->addJavascript(pongho_url('/vendor/TinyMCE/tinymce.min.js'))
            ->addJavascriptInline(
                <<<JS
    Pongho.wysiwyg = {
        options: $options
    };

    /**
     * Pongho.wysiwyg.cleanUrls()
     *
     * Serve a mantenere inalterati gli shortcode inseriti negli attributi href e src dei tag
     * ATTENZIONE: gli url relativi devono essere inseriti con / iniziale altrimenti non vengono convertiti
     *
     * Dati i link
     * <p><a href="[config key='company_website' /]">url con shortcode [</a></p>
     * <p><a href="{COMPANY_WEBSITE}">url con shortcode { (non viene parsato dai nodi)</a></p>
     * <p><a href="/prova-url-shortcode/">url relativo</a></p>
     * <p><a href="http://www.google.it">url assoluto</a></p>
     *
     * L'output risulta in
     * <p><a href="[config key='company_website' /]">url con shortcode [</a></p>
     * <p><a href="{COMPANY_WEBSITE}">url con shortcode { (non viene parsato dai nodi)</a></p>
     * <p><a href="http://www.domain.it/prova-url-shortcode/">url relativo</a></p>
     * <p><a href="http://www.google.it">url assoluto</a></p>
     */
    Pongho.wysiwyg.cleanUrls = function (editor) {
        // http://www.tinymce.com/wiki.php/api4:method.tinymce.Editor.convertURL
        editor.originalConvertUrl = editor.convertURL;

        editor.convertURL = function (url, name, elm) {
            if (typeof url === 'string') {
                if (url[0] === '/' && url[1] != '/') {
                    url = '{$host_url}' + url;
                }
            }

            return editor.originalConvertUrl(url, name, elm);
        };
    };

    /**
     * Pongho.wysiwyg.fixInitialContent()
     */
    Pongho.wysiwyg.fixInitialContent = function (editor) {
        editor.on('init', function() {
            var doc = $(editor.dom.doc);

            doc.find('table').each(function () {
                var table = $(this);

                if (table.attr('width')) {
                    table.css('width', table.attr('width'));
                }

                if (table.attr('bgcolor')) {
                    table.css('background-color', table.attr('bgcolor'));
                }

                if (table.attr('align') === 'center') {
                    table.css({
                        marginLeft: 'auto',
                        marginRight: 'auto'
                    });
                }
            });

            doc.find('td, th').each(function () {
                var td = $(this);

                if (td.attr('width')) {
                    td.css('width', td.attr('width'));
                }

                if (td.attr('bgcolor')) {
                    td.css('background-color', td.attr('bgcolor'));
                }

                if (td.attr('align')) {
                    td.css('text-align', td.attr('align'));

                    td.children('table').css({
                        marginLeft: 'auto',
                        marginRight: 'auto'
                    });
                }
            });
        });
    };

    /**
     * Pongho.wysiwyg.setup()
     */
    Pongho.wysiwyg.setup = function (editor) {
        Pongho.wysiwyg.cleanUrls(editor);
        Pongho.wysiwyg.fixInitialContent(editor);
    };

    Pongho.wysiwyg.options.normal.setup = Pongho.wysiwyg.setup;
    Pongho.wysiwyg.options.small.setup = Pongho.wysiwyg.setup;
    Pongho.wysiwyg.options.full.setup = Pongho.wysiwyg.setup;
    Pongho.wysiwyg.options.newsletter.setup = Pongho.wysiwyg.setup;
JS
            );
    }

    /**
     * Abilita il supporto per il campo Mappa.
     */
    protected function enableGeoField()
    {
        $this->getHelper()->getHead()
            ->addJavaScript(pongho_url('/Application/Core/Resources/views/js/geo-localization.js'));
    }

    /**
     * @return AssociativeArrayObject
     */
    public function convertTableColumns()
    {
        $columns = $this->getTableColumns();

        // Questa parte la fa per retro compatibilità
        if (!is_hash($columns)) {
            $cols = array();
            foreach ($columns as $column) {
                $cols[$column['name']] = $column;
                unset($column['name']);
            }

            $columns = $cols;
        }

        $columns = $this->getHelper()->filter($this, $this->getEventNamespace('filter_table_columns'), new AssociativeArrayObject($columns));

        return $columns->getArrayCopy();
    }

    /**
     * @return string
     */
    protected function getTableEmptyRecordsetText()
    {
        $in_search = $this->getRequest()->query->get('q', '') !== ''
            || $this->getRequest()->query->get('f')
            || $this->getRequest()->query->get('filter');

        if ($in_search) {
            return $this->getArchiveSearchNoResultsText();
        }

        if ($this->canAdd()) {
            $path = '/' . $this->getParameter('path') . '/';

            return '<a href="' . $this->url($path . 'add/') .'">' . $this->getArchiveAddFirstText() .'</a>';
        }

        return $this->getArchiveEmptyRecordsetText();
    }

    /**
     * Restituisce il gestore degli archivi
     *
     * @param array $config
     * @return ArchiveManager
     */
    protected function getArchiveManager(array $config = null)
    {
        if ($this->am === null) {
            $path = '/' . $this->getParameter('path') . '/';

            if ($config === null) {
                $config = array(
                    'model_class'   => $this->getModelClass(),
                    'model_options' => $this->getHelper()->filter($this, $this->getEventNamespace('filter_model_options'), $this->getModelOptions()),
                    'model_fields'  => $this->getModelFieldsConfigWrapper(),
                    'parse_row'     => array($this, 'parseArchiveRow'),
                    'search_fields' => $this->getSearchFieldsWrapper(),
                    'fast_filters'  => $this->getFastFilters(),
                    'form'          => array($this, 'getFiltersForm'),
                    'table'         => array(
                        'columns'         => array($this, 'convertTableColumns'),
                        'empty_recordset' => $this->getTableEmptyRecordsetText(),
                    ),
                );
            }

            $this->am = new ArchiveManager(
                $this->getRequest(),
                $this->getContainer()->getService('translator'),
                $this->getHelper()->filter($this, $this->getEventNamespace('filter_archive_config'), $config),
                pongho_url($path),
                $this->page
            );
        }

        return $this->am;
    }

    /**
     * Aggiunge le azioni alle righe dell'elenco dei modelli
     *
     * @param $row
     */
    public function parseArchiveRow($row)
    {
        $path = $this->getParameter('path');

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $actions = array(
            array($translator->trans('Edit'), 'href' => $this->url("/{$path}/edit/{$row->id}/"), 'class' => 'edit'),
            array($translator->trans('Delete'), 'href' => $this->url("/{$path}/delete/{$row->id}/"), 'class' => 'delete'),
        );

        $row->actions = $this->parseActions($actions);
    }

    /**
     * Aggiunge le azioni alle righe dell'elenco dei filtri salvati
     *
     * @param $row
     *
     * @return string
     */
    public function parseFilterRow($row)
    {
        $path = $this->getParameter('path');

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $actions = array(
            array($translator->trans('Execute'), 'href' => $this->url("/{$path}/?filter={$row->id}"), 'class' => 'run'),
            array($translator->trans('Delete'), 'href' => $this->url("/{$path}/filterdelete/{$row->id}/"), 'class' => 'delete'),
        );
        $row->actions = $this->parseActions($actions);
    }

    /**
     * @return array
     */
    protected function getSearchFieldsWrapper()
    {
        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_search_fields'), $this->getSearchFields());
    }

    /**
     * @return array
     */
    protected function getFilterEnumValuesWrapper()
    {
        return $this->getHelper()->filter($this, $this->getEventNamespace('filters.filter_enums_values'), $this->getFilterEnumValues());
    }

    /**
     * @return array
     */
    protected function getFilterSelectValuesWrapper()
    {
        return $this->getHelper()->filter($this, $this->getEventNamespace('filters.filter_select_values'), $this->getFilterSelectValues());
    }

    /**
     * @return array
     */
    protected function getModelFieldsConfigWrapper()
    {
        return $this->getHelper()->filter($this, $this->getEventNamespace('filters.filter_model_fields'), $this->getModelFieldsConfig());
    }

    /**
     * Restituisce la form dei filtri
     *
     * @param \Application\Admin\Model\Filter $filter
     *
     * @return \Pongho\Form\Form
     */
    public function getFiltersForm(Filter $filter)
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $path = $this->getParameter('path');

        $subject = new FilterSubject($filter);

        $actions_view = new View(__DIR__ . '/../Resources/views/filter-actions.php');

        $actions_view->assignVar('translator', $translator);

        $config = new FormConfig('filters', $subject, $translator);

        $config->addFields(
            '',
            array(
                'explain' => array(
                    'class' => 'Pongho\\Form\\Fieldset',
                    'description' => $translator->trans('The filters will meet all the conditions. If a condition will be selected multiple times, it will be taken into account only the last occurrence'),
                    'attributes' => array(
                        'class' => 'filters-description',
                    ),
                    'settings' => array(
                        'hide_label' => true,
                    ),
                ),
                'filter' => array(
                    'class'      => 'Pongho\\Form\\Repeater\\SerializableRepeater',
                    'subject'    => $subject,
                    'attributes' => array('class' => 'panel'),
                    'label'      => $translator->trans('Filters'),
                    'settings'   => array(
                        'check_valid_row_callback' => array($this, 'filterCheckValidRowCallback'),
                        'filter_fields_config'     => array($this, 'filterFiltersFieldsCallback'),
                        'row_settings'             => array(
                            'title'        => 'Cerca per:',
                            'open-label'   => $translator->trans('Advanced'),
                            'close-label'  => $translator->trans('Close'),
                            'delete-label' => $translator->trans('Delete'),
                            'hide-open'    => true,
                            'show-delete'  => true,
                            'show-content' => false,
                            'hide-options' => true,
                        ),
                        'columns'                  => $this->getModelFields(),
                    ),
                ),
                'actions' => array(
                    'class'      => 'Pongho\\Form\\Fieldset',
                    'label'      => $translator->trans('Save filter'),
                    'attributes' => array('class' => 'filters-actions'),
                    'settings'   => array(
                        'hide_label' => true,
                        'view'       => $actions_view,
                    ),
                ),
            )
        );

        $config->addField(
            'filter/main',
            array(
                'class'  => 'Application\\Admin\\Form\\Repeater\\RowMain',
            )
        );

        $config->addFields(
            'filter/main',
            array(
                'field'          => array(
                    'class'      => 'Pongho\\Form\\Field\\SelectField',
                    'options'    => $this->getModelFieldsSelectOptions(),
                    'attributes' => array(
                        'class'    => 'filter-field chosen',
                        'data-ref' => pongho_url('/' . $path . '/filtervalue/'),
                    ),
                ),
                'operator'       => array(
                    'class'      => 'Pongho\\Form\\Field\\SelectField',
                    'options'    => array(),
                    'attributes' => array(
                        'class' => 'filter-operator',
                    ),
                ),
                'type'           => array(
                    'class'      => 'Pongho\\Form\\Field\\HiddenField',
                    'attributes' => array(
                        'class' => 'filter-type',
                    ),
                ),
                'valuefield'     => array(
                    'class'      => 'Pongho\\Form\\Field\\HiddenField',
                    'attributes' => array(
                        'class' => 'filter-valuefield',
                    ),
                ),
                'value_text'     => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'attributes' => array(
                        'placeholder' => $translator->trans('Empty'),
                        'class'       => 'input_text filter-value filter-value-text',
                    ),
                ),
                'value_select'   => array(
                    'class'      => 'Pongho\\Form\\Field\\SelectField',
                    'options'    => array(),
                    'attributes' => array(
                        'class' => 'filter-value filter-value-select',
                        'style' => 'display: none;',
                    ),
                ),
                'value_multiple' => array(
                    'class'      => 'Pongho\\Form\\Field\\SelectField',
                    'options'    => array(),
                    'attributes' => array(
                        'class'    => 'filter-value filter-value-multiple',
                        'multiple' => 'multiple',
                        'style'    => 'display: none;',
                        'size'     => 10,
                    ),
                ),
                'datewrapper' => array(
                    'class' => 'Pongho\\Form\\Fieldset',
                    'attributes' => array(
                        'class' => 'picker-wrapper date-wrapper',
                    ),
                    'settings' => array(
                        'view' => new ViewClosure(array($this, 'getWrapperView'))
                    ),
                ),
                'datetimewrapper' => array(
                    'class' => 'Pongho\\Form\\Fieldset',
                    'attributes' => array(
                        'class' => 'picker-wrapper datetime-wrapper',
                    ),
                    'settings' => array(
                        'view' => new ViewClosure(array($this, 'getWrapperView'))
                    ),
                ),
                'timewrapper' => array(
                    'class' => 'Pongho\\Form\\Fieldset',
                    'attributes' => array(
                        'class' => 'picker-wrapper time-wrapper',
                    ),
                    'settings' => array(
                        'view' => new ViewClosure(array($this, 'getWrapperView'))
                    ),
                ),
            )
        );

        $config->addFields(
            'filter/main/datewrapper',
            array(
                'value_date'     => array(
                    'class'      => 'Pongho\\Form\\Field\\DateField',
                    'attributes' => array(
                        'class'       => 'filter-value filter-value-date filter-value-date-between date-between-from',
                        'placeholder' => $translator->trans('Empty'),
                        'style'       => 'display: none;',
                    ),
                ),
                'value_date_to'     => array(
                    'class'      => 'Pongho\\Form\\Field\\DateField',
                    'attributes' => array(
                        'class'       => 'filter-value filter-value-date-between date-between-to',
                        'placeholder' => $translator->trans('Empty'),
                        'style'       => 'display: none;',
                    ),
                ),
            )
        );

        $config->addFields(
            'filter/main/datetimewrapper',
            array(
                'value_datetime'     => array(
                    'class'      => 'Pongho\\Form\\Field\\DateTimeField',
                    'attributes' => array(
                        'class'       => 'filter-value filter-value-datetime filter-value-datetime-between datetime-between-from',
                        'placeholder' => $translator->trans('Empty'),
                        'style'       => 'display: none;',
                        'data-options' => htmlspecialchars(
                            json_encode(
                                array(
                                    'stepMinute' => 1,
                                )
                            )
                        ),
                    ),
                ),
                'value_datetime_to'     => array(
                    'class'      => 'Pongho\\Form\\Field\\DateTimeField',
                    'attributes' => array(
                        'class'       => 'filter-value filter-value-datetime-between datetime-between-to',
                        'placeholder' => $translator->trans('Empty'),
                        'style'       => 'display: none;',
                        'data-options' => htmlspecialchars(
                            json_encode(
                                array(
                                    'stepMinute' => 1,
                                )
                            )
                        ),
                    ),
                ),
            )
        );

        $config->addFields(
            'filter/main/timewrapper',
            array(
                'value_time'     => array(
                    'class'      => 'Pongho\\Form\\Field\\TimeField',
                    'attributes' => array(
                        'class'       => 'filter-value filter-value-time filter-value-time-between time-between-from',
                        'placeholder' => $translator->trans('Empty'),
                        'style'       => 'display: none;',
                        'data-options' => htmlspecialchars(
                            json_encode(
                                array(
                                    'showSecond' => 1,
                                    'stepMinute' => 1,
                                    'timeFormat' => 'HH:mm:ss',
                                )
                            )
                        ),
                    ),
                ),
                'value_time_to'     => array(
                    'class'      => 'Pongho\\Form\\Field\\TimeField',
                    'attributes' => array(
                        'class'       => 'filter-value filter-value-time-between time-between-to',
                        'placeholder' => $translator->trans('Empty'),
                        'style'       => 'display: none;',
                        'data-options' => htmlspecialchars(
                            json_encode(
                                array(
                                    'showSecond' => 1,
                                    'stepMinute' => 1,
                                    'timeFormat' => 'HH:mm:ss',
                                )
                            )
                        ),
                    ),
                ),
            )
        );

        $config->addFields(
            'actions',
            array(
                'name' => array(
                    'class'      => 'Pongho\\Form\\Field\\TextField',
                    'attributes' => array('class' => 'input_text', 'placeholder' => $translator->trans('Name')),
                    'settings'   => array(
                        'hide_label' => true,
                    ),
                ),
            )
        );

        // Aggiungo il campo ID per sovrascrivere un filtro salvato
        if (!$filter->isNewRecord()) {
            $config->addField(
                'actions/id',
                array(
                    'class' => 'Pongho\\Form\\Field\\HiddenField',
                )
            );
        }

        $config = $this->getHelper()->filter($this, $this->getEventNamespace('filter_filters'), $config);

        $form = FormBuilder::buildConfig($config)
            ->setGenerator(new AdminGenerator($this->getHelper()->getLocalization()))
            ->setAttribute('action', $this->url('/' . $path . '/'))
            ->setAttribute('data-ref', $this->url('/' . $path . '/filterslist/'))
            ->setAttribute('data-save', $this->url('/' . $path . '/' . ($filter->isNewRecord() ? '' : '?filter=' . $filter->id)));
        $form->afterAddFieldToParent();

        return $form;
    }

    /**
     * @param Fieldset $field
     * @return string
     */
    public function getWrapperView(Fieldset $field)
    {
        $tags = '';

        /** @var \Pongho\Form\BaseField $f */
        foreach ($field->getFields() as $f) {
            $tags .= $field->getGenerator()->renderTag($f);
        }

        return '<div style="display: inline-block;" class="' . implode(' ', $field->getCssClasses()) . '">' . $tags . '</div>';
    }

    /**
     * Controlla la validità dei singoli filtri prima di salvarli
     *
     * @param       $id
     * @param array $row_request
     *
     * @return bool
     */
    public function filterCheckValidRowCallback($id, array $row_request)
    {
        if (!isset($row_request['field']) || $row_request['field'] === '') {
            return false;
        }

        return true;
    }

    /**
     * Gestisce il contenuto delle righe dei singoli filtri
     *
     * @param \Pongho\Form\Repeater\RepeaterRow $field
     * @param array                             $row_fields
     * @return array
     */
    public function filterFiltersFieldsCallback(RepeaterRow $field, array $row_fields)
    {
        $key = $field->getKey();
        /** @var \Pongho\Form\Subject\RowSubjectInterface $row_subject */
        $row_subject = $field->getsubject();

        if ($key !== 'new') {
            $config = FormConfig::fromArray($row_fields);

            $operators = $this->getHelper()->filter($this, $this->getEventNamespace('filter_filters_operators'), $this->getArchiveManager()->getOperators());

            $operator_select = $config->getField('main/operator');
            $operator_select['options'] = isset($operators[$row_subject->get('type')]) ? $operators[$row_subject->get('type')] : array();

            if (($row_subject->get('type') === 'foreign') || ($row_subject->get('type') === 'enum')) {
                $callback = array(
                    $this,
                    ($row_subject->get('type') === 'enum' ? 'getFilterRowEnumOptionsCallback' : 'getFilterRowSelectOptionsCallback')
                );
                $options = call_user_func($callback, $row_subject->get('field'), $row_subject);

                $valuefield = $row_subject->get('valuefield');
                $value_select = $config->getField('main/' . $valuefield);
                $value_select['options'] = $options;
            }

            return $config->getformFields();
        }

        return $row_fields;
    }

    /**
     * Restituisce i campi del modello tipizzati per gestire gli operatori dei filtri in base al tipo delle colonne del modello
     *
     * @throws \LogicException
     * @return array
     */
    protected function getModelFields()
    {
        $fields = $this->getModelFieldsConfigWrapper();

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $model_fields = array(null => $translator->trans('Select a field'));

        /** @var \ActiveRecord\Base $model_class */
        $model_class = $this->getModelClass();
        $model_columns = $model_class::columns();

        foreach ($fields as $field => $options) {
            if (isset($options['data-type'])) {
                $filter_type = $options['data-type'];
            } elseif (isset($options['type'])) {
                $filter_type = $options['type'];
            } else {
                $filter_type = '';
                if (array_key_exists($field, $model_columns)) {
                    switch ($model_columns[$field]->type) {
                        case Column::STRING :
                            $filter_type = 'string';
                            break;

                        case Column::BOOLEAN:
                            $filter_type = 'enum';
                            break;

                        case Column::INTEGER:
                        case Column::DECIMAL:
                            $filter_type = 'numeric';
                            break;

                        case Column::DATE:
                        case Column::DATETIME:
                            $filter_type = 'date';
                            break;
                    }
                } else {
                    throw new \LogicException(sprintf('The column "%s" is not defined in the table "%s" (maybe is it a join?), therefore you must manually define the "data-type"', $field, $model_class::tableName()));
                }
            }

            $model_fields[$field] = array(
                isset($options['label']) ? $options['label'] : $field,
                'data-type' => $filter_type
            );

            if (isset($options['attributes'])) {
                unset($options['attributes']['data-type']);

                foreach ($options['attributes'] as $attribute => $value) {
                    $model_fields[$field][$attribute] = $value;
                }
            }

            if (isset($options['optgroup'])) {
                $model_fields[$field]['optgroup'] = $options['optgroup'];
            }
        }

        return $model_fields;
    }

    /**
     * @return array
     */
    protected function getModelFieldsSelectOptions()
    {
        $options = array();

        foreach ($this->getModelFields() as $field => $config) {
            if (is_array($config) && isset($config['optgroup'])) {
                $optgroup = $config['optgroup'];
                unset($config['optgroup']);
                $options[$optgroup]['optgroup'][$field] = $config;
            } else {
                $options[$field] = $config;
            }
        }

        return $options;
    }

    /**
     * Restituisce le opzioni per i campi enum dei filtri
     *
     * @param      $field_name
     *
     * @return array
     */
    public function getFilterRowEnumOptionsCallback($field_name)
    {
        $enums = $this->getFilterEnumValuesWrapper();

        $options = array();
        if (array_key_exists($field_name, $enums)) {
            $options = $enums[$field_name];
        }

        return $options;
    }

    /**
     * Gestisce la generazione delle opzioni della select per i dati del campo del modello
     *
     * @param      $field_name
     *
     * @return array
     */
    public function getFilterRowSelectOptionsCallback($field_name)
    {
        /** @var \ActiveRecord\Base $model_class */
        $model_class = $this->getModelClass();

        $fields = array(
            'default' => function ($field) use ($model_class) {
                    $options = array(
                        'select' => $field,
                        'group'  => $field,
                        'order'  => $field . ' ASC',
                    );

                    $o = array();

                    foreach ($model_class::all($options) as $row) {
                        $o[$row->$field] = $row->$field;
                    }

                    return $o;
                },
        );

        $fields = array_merge($fields, $this->getFilterSelectValuesWrapper());

        if (array_key_exists($field_name, $fields)) {
            $options = call_user_func($fields[$field_name]);
        } else {
            $options = call_user_func($fields['default'], $field_name);
        }

        return $options;
    }

    /**
     * Restituisce le opzioni per la select dei valori del filtro
     *
     * @access public
     *
     * @return \Pongho\Http\JsonResponse
     */
    public function filtervalueAction()
    {
        $field = $this->getRequest()->query->get('f');

        return new JsonResponse($this->getFilterRowSelectOptionsCallback($field));
    }

    /**
     * Esegue le azioni legate al filtro
     *
     * @access public
     * @deprecated
     *
     * @example
     * -> GET
     * /pongho/users/filter/        => form vuota
     * /pongho/users/filter/123/    => carico il filtro 123 da db
     * -> POST
     * /pongho/users/filter/        => aggiungo un nuovo filtro con i dati provenienti da POST
     * /pongho/users/filter/123/    => modifico il filtro 123 con i dati provenienti da POST
     */
    public function filterAction()
    {
        throw new HttpNotFoundException('This action has been deprecated, use "/pongho/path/?filter=123" instead');
    }

    /**
     * Restituisce l'elenco dei filtri salvati
     *
     * @access public
     */
    public function filtersAction()
    {
        return new Response($this->getFiltersListHtml());
    }

    /**
     * Restituisce i filtri per l'elenco corrente
     *
     * @return Filter[]
     */
    protected function getFiltersList()
    {
        $filters = Filter::all(
            array(
                'conditions' => array(
                    'namespace = :namespace AND (user_id = :user OR user_id = :anon)',
                    'namespace' => $this->getNamespace(),
                    'user'      => $this->getContainer()->getService('current_user')->id,
                    'anon'      => Account::ANONYMOUS,
                )
            )
        );

        /** @var $filter \Application\Admin\Model\Filter */
        foreach ($filters as $filter) {
            $this->parseFilterRow($filter);
        }

        return $filters;
    }

    /**
     * Restituisce il template dell'elenco dei filtri salvati
     *
     * @return string
     */
    protected function getFiltersListHtml()
    {
        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../Resources/views/list_filters_table.php')
            ->assignVars(
                array(
                    'filters_list' => $this->getFiltersList(),
                )
            );

        return $this->getHelper()->getBodyView()->render();
    }

    /**
     * Elimina un filtro salvato
     *
     * @access public
     */
    public function filterdeleteAction()
    {
        $id = intval($this->getParameter('id'));

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        /** @var $filter \Application\Admin\Model\Filter */
        if (($filter = Filter::find($id)) !== null) {
            if (!$filter->delete()) {
                return new JsonResponse(array(
                    'error'   => true,
                    'message' => $translator->trans('An error occurred when trying to delete the filter')
                ));
            }
        }

        return new JsonResponse(array('message' => $translator->trans('The filter has been deleted')));
    }

    /**
     * Gestisce le azioni di massa dei filtri
     *
     * @access public
     */
    public function filtermassAction()
    {

    }

    /**
     * Metodo da sovrascrivere per aggiungere alla vista variabili richieste dai template personalizzati per le singole applicazioni
     *
     * @return null
     */
    protected function beforeRenderArchive()
    {
    }

    /**
     * Restituisce il codice per la parte sinistra dell'header di amministrazione
     *
     * @return string
     */
    protected function getLeftHeaderCode()
    {
        return $this->getAddButtonCode() . $this->getFastFiltersCode();
    }

    /**
     * Restituisce il codice per la parte destra dell'header di amministrazione
     *
     * @return string
     */
    protected function getRightHeaderCode()
    {
        return $this->getSearchFormCode() . $this->getFiltersButtonCode();
    }

    /**
     * Restituisce il codice per il pulsante di aggiunta di un nuovo record
     *
     * @return string
     */
    protected function getAddButtonCode()
    {
        $code = '';

        $path = $this->getParameter('path');

        if ($this->hasAddButton()) {
            $code .= '<div class="actions"><a href="' . $this->url('/' . $path . '/add/') . '" class="primary pongho-button">' . $this->getArchiveAddButtonText() . '</a></div>';
        }

        return $code;
    }

    /**
     * Restituisce il codice per i filtri rapidi
     *
     * @return string
     */
    protected function getFastFiltersCode()
    {
        $code = '';

        if ($this->getArchiveManager()->isFastFiltersEnabled()) {
            $code .= '<div class="filter-status"><ul>' . $this->getArchiveManager()->getFastFilters() . '</ul></div>';
        }

        return $code;
    }

    /**
     * Restituisce il codice per la Form di ricerca
     *
     * @return string
     */
    protected function getSearchFormCode()
    {
        $code = '';

        $path = $this->getParameter('path');
        $search_url = $this->url('/' . $path . '/');
        $search_query = $this->getRequest()->query->get('q', '');
        $site = $this->getHelper()->getSite();

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        if ($this->getArchiveManager()->isSearchEnabled()) {
            $code .= <<<HTML
    <form id="search-form" style="float:left;" method="get" class="pongho-form" action="{$search_url}">
        <input type="hidden" name="site" value="{$site->id}"/>
        <input type="text" id="search-query" name="q" class="input_text" value="{$search_query}"/>
        <input type="submit" id="search-button" class="pongho-button" value="{$translator->trans('Search')}">
    </form>
HTML;
        }

        return $code;
    }

    /**
     * Restituisce il codice per il bottone di apertura dei filtri
     *
     * @return string
     */
    protected function getFiltersButtonCode()
    {
        $code = '';

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        if ($this->getArchiveManager()->isFiltersEnabled()) {
            $code .= '<button id="filter-toggle" class="pongho-button">' . $translator->trans('Filters') . '</button>';
        }

        return $code;
    }

    /**
     * Form dei filtri
     *
     * @param Form $form
     * @return string
     */
    protected function getFiltersCode(Form $form)
    {
        $view = new View(__DIR__ . '/../Resources/views/list_filters.php');
        $view->assignVars(
            array(
                'filters_list' => $this->getFiltersList(),
                'filters'      => $form->render(),
            )
        );

        return $view->render();
    }

    /**
     * Form della tabella
     *
     * @return string
     */
    protected function getTableCode()
    {
        $view = new View($this->getArchiveTableTemplatePath());

        $view->assignVars(
            array(
                'table'               => $this->getArchiveManager()->getTable(),
                'mass_action_url'     => $this->getMassActionUrl(),
                'mass_action_buttons' => $this->getMassActionButtons(),
                'pagination'          => $this->getArchiveManager()->getPagination(),
            )
        );

        return $view->render();
    }
}
