<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele De Nobili
 * @package Application\Admin
 */

namespace Application\Admin\Controller;

use Application\Core\BaseController;
use Pongho\Core\TemplateResponse;
use Pongho\Http\Exception\HttpException;
use Pongho\Http\Exception\HttpUnauthorizedException;

/**
 * Controller base per l’amministrazione.
 */
abstract class AdminController extends BaseController
{
    /**
     * {@inheritdoc}
     */
    protected function handleInvalidResponse($response)
    {
        return new TemplateResponse($this->getHelper()->getView(), $this->getHelper()->getResponseHeaders());
    }

    /**
     * Numero di record per pagina da visualizzare negli elenchi.
     *
     * @var integer
     */
    protected $records_per_page = 50;

    /**
     * {@inheritdoc}
     */
    protected function boot()
    {
        parent::boot();

        // Utente loggato?
        if (!$this->getHelper()->getUser()->isLogged()) {
            /** @var \Application\Core\Model\Site $main_site */
            $main_site = $this->getContainer()->getService('main_site');

            $qs = http_build_query(array('redirect' => $this->currentUrl('&')), null, '&');
            $login_url = $main_site->domain() . '/user/login/?' . $qs;

            // Solo le azioni possono restituire risposte.
            // Quindi  la imposto.
            $this->setResponse($this->getHelper()->redirectResponse($login_url));

            return;
        }

        // Permesso consentito?
        if (!$this->getHelper()->getUser()->hasPermit('admin')) {
            throw new HttpUnauthorizedException();
        }

        // Aggiorno i servizi nella dependency injection
        $this->container->setService('template_engine', $this->getHelper()->getBodyView());
        $this->container->setService('admin_view_header', $this->getHelper()->getHeaderView());
        $this->container->setService('admin_view_body', $this->getHelper()->getBodyView());
        $this->container->setService('admin_view_footer', $this->getHelper()->getFooterView());

        // css e js
        $this->getHelper()->getHead()
            ->setTitle($this->getContainer()->getService('translator')->trans('Admin panel'))
            ->resetCss()
            ->addCss(pongho_url('/Application/Admin/Resources/views/css/style.css?v=' . filemtime(__DIR__ . '/../Resources/views/css/style.css')))
            ->addCss(pongho_url('/vendor/jQueryUI/ui-lightness/jquery-ui.min.css?v=' . filemtime(__DIR__ . '/../../../vendor/jQueryUI/ui-lightness/jquery-ui.min.css')))
            ->addCss(pongho_url('/vendor/jQueryUI-widgets/timepicker/jquery-ui-timepicker-addon.css?v=' . filemtime(__DIR__ . '/../../../vendor/jQueryUI-widgets/timepicker/jquery-ui-timepicker-addon.css')))
            ->addCss(pongho_url('/vendor/jQueryUI-widgets/Spectrum/spectrum.css?v=' . filemtime(__DIR__ . '/../../../vendor/jQueryUI-widgets/Spectrum/spectrum.css')))
            ->addCss(pongho_url('/vendor/jQuery-plugins/Chosen/chosen.min.css?v=' . filemtime(__DIR__ . '/../../../vendor/jQuery-plugins/Chosen/chosen.min.css')))
            ->resetJavaScript()
            ->addJavaScriptInline($this->getJavascriptInline())
            ->addJavaScript(pongho_url('/Application/Admin/Resources/views/js/onload.php'));

        // sidebar
        $this->displaySidebar();
    }

    /**
     * {@inheritdoc}
     */
    public function handleRaw($action_method)
    {
        try {
            return parent::handleRaw($action_method);
        } catch (HttpException $e) {
            switch ($e->getStatusCode()) {
                case 401:
                case 404:
                    $template = __DIR__ . '/../Resources/views/' . $e->getStatusCode() . '.php';
                    break;

                default:
                    $template = __DIR__ . '/../Resources/views/400.php';
                    $this->getHelper()->getBodyView()->assignVar('exception', $e);
                    break;

            }

            $this->getHelper()->getBodyView()->setTemplatePath($template);

            return new TemplateResponse($this->getHelper()->getView());
        }
    }

    /**
     * Restituisce l'helper per l'admin.
     *
     * @return \Application\Admin\Utilities\AdminControllerHelper
     */
    public function getHelper()
    {
        return $this->container->getService('admin_controller_helper');
    }

    /**
     * {@inheritdoc}
     *
     * @deprecated
     */
    public function getView()
    {
        return $this->getHelper()->getView();
    }

    /**
     * @return \Pongho\Template\ViewInterface
     *
     * @deprecated
     */
    public function getHeaderView()
    {
        return $this->getHelper()->getHeaderView();
    }

    /**
     * @return \Pongho\Template\ViewInterface
     *
     * @deprecated
     */
    public function getBodyView()
    {
       return $this->getHelper()->getBodyView();
    }

    /**
     * @return \Pongho\Template\ViewInterface
     *
     * @deprecated
     */
    public function getFooterView()
    {
        return $this->getHelper()->getFooterView();
    }

    /**
     * @deprecated
     */
    protected function getViewGlobalVariables()
    {
        return $this->getHelper()->getViewGlobalVariables();
    }

    /**
     *
     */
    protected function getJavascriptInline()
    {
        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->getContainer()->getService('translator');

        $pongho = array(
            'action'      => $this->getAction(),
            'apiUrl'      => url('/api'),
            'maxFileSize' => 2000000,
            'lang'        => array(
                'fileUploadErrors' => array(
                    'maxFileSize'      => $translator->trans('The file size is too big'),
                    'minFileSize'      => $translator->trans('The file size is too small'),
                    'acceptFileTypes'  => $translator->trans('Extension not allowed'),
                    'maxNumberOfFiles' => $translator->trans('The total number of files exceed the limit'),
                    'uploadedBytes'    => $translator->trans('The total size of the files exceed the limit'),
                    'emptyResult'      => $translator->trans('The file looks empty'),
                ),
            ),
            'chosen'      => array(
                'include_group_label_in_selected' => true,
                'no_results_text'                 => $translator->trans('No results match'),
                'placeholder_text_multiple'       => $translator->trans('Select Some Options'),
                'placeholder_text_single'         => $translator->trans('Select an Option'),
            ),
        );

        $this->getHelper()->filter($this, 'admin.filter_js_pongho_var', $pongho);

        $pongho = json_encode($pongho);
        $code = <<<JS
var tmp = {$pongho},
    Pongho = window.Pongho || {};

for (var property in tmp) {
    if (tmp.hasOwnProperty(property)) {
        Pongho[property] = tmp[property];
    }
}
JS;

        return $code;
    }

    /**
     * Visualizza la sidebar
     */
    protected function displaySidebar()
    {
        $this->getHelper()->getFooterView()
            ->assignVar('sidebar_menu', $this->getContainer()->getService('admin_sidebar'));
    }

    /**
     * Restituisce il modulo amministrativo.
     *
     * @return \Application\Admin\Model\AdminModule
     *
     * @deprecated
     */
    public function getAdminModule()
    {
        return $this->getParameter('admin_module');
    }

    /**
     * Restituisce l’ID del modulo amministrativo.
     *
     * @return integer
     *
     * @deprecated
     */
    public function getAdminModuleId()
    {
        return $this->getParameter('admin_module')->id;
    }

    /**
     * Restituisce l’ID del modulo.
     *
     * @return integer
     */
    public function getModuleId()
    {
        return $this->getParameter('module_id');
    }

    /**
     * Restituisce il modello ModuleSite corrente.
     *
     * @param boolean $thrown
     * @return \Application\Core\Model\ModuleSite
     * @throws \UnexpectedValueException Quando il modulo amministrativo non è associato ad alcun modulo.
     *
     * @deprecated
     */
    public function getSiteModule($thrown = true)
    {
        return $this->getHelper()->getModuleSite($thrown);
    }

    /** @var int */
    protected $content_language;

    /**
     * Restituisce la lingua per l'interfaccia
     * @return int
     *
     * @deprecated
     */
    public function getContentLanguageId()
    {
        /** @var \Application\Admin\Utilities\ContentLanguage $content_language */
        $content_language = $this->getContainer()->getService('content_language');

        return $content_language->getId();
    }

    /**
     * {@inheritdoc}
     *
     * @deprecated
     */
    protected function getSiteModuleClassName()
    {
        return $this->getHelper()->getModuleSiteClass();
    }

    /**
     * Estende la funzione pongho_url() per aggiungere la query con il sito e la lingua corrente.
     *
     * Se questo metodo viene usato assieme alla risposta `Pongho\Http\RedirectResponse` si consiglia di usare il
     * carattere `&` come argomento per `$arg_separator`.
     *
     * @param string $url           Indirizzo da elaborare.
     * @param string $arg_separator Separatore dei valori.
     * @return string
     * @throws \InvalidArgumentException Se l’indirizzo passato contiene il prefisso `/pongho/`
     */
    public function url($url, $arg_separator = '&amp;')
    {
        /** @var \Application\Admin\Utilities\PonghoUrl $pongho_url */
        $pongho_url = $this->getContainer()->getService('pongho_url');

        return $pongho_url->url($url, $arg_separator);
    }

    /**
     * Restituisce l'url corrente.
     *
     * @param string $arg_separator
     * @return string
     */
    public function currentUrl($arg_separator = '&amp;')
    {
        /** @var \Application\Admin\Utilities\PonghoUrl $pongho_url */
        $pongho_url = $this->getContainer()->getService('pongho_url');

        return $pongho_url->currentUrl($arg_separator);
    }
}
