<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Admin
 */

namespace Application\Admin\Utilities;

use Pongho\Utilities\Inflector;

class Panel implements \Stringable
{
    protected $name;

    protected $label;

    protected $content;

    protected $prefix = 'tab';

    protected $link;

    protected $active;

    protected $is_ajax = false;

    /**
     * @param string $name
     * @param string $label
     * @throws \LogicException
     */
    public function __construct($name, $label, array $options = [])
    {
        $this->setTabName($name);
        $this->setTabLabel($label);

        if (array_key_exists('isAjax', $options)) {
            $this->is_ajax = $options['isAjax'];
        }

        if (array_key_exists('content', $options)) {
            $this->setContent($options['content']);

            if (array_key_exists('link', $options)) {
                $this->setLink($options['link']);
            }

            if (array_key_exists('prefix', $options)) {
                $this->setPrefix($options['prefix']);
            }
        } elseif (array_key_exists('link', $options)) {
            $this->setLink($options['link']);
        } else {
            throw new \LogicException('You can\'t create an empty "' . $name . '" tab!');
        }
    }

    /**
     * Restituisce il nome della tab
     *
     * @return string
     */
    public function getTabName()
    {
        return $this->name;
    }

    /**
     * Imposta il nome della tab
     *
     * @param string $name
     */
    public function setTabName($name)
    {
        $this->name = Inflector::selectorize($name);
    }

    /**
     * Restituisce la label della tab
     *
     * @return string
     */
    public function getTabLabel()
    {
        return $this->label;
    }

    /**
     * Imposta la label della tab
     *
     * @param string $label
     */
    public function setTabLabel($label)
    {
        $this->label = $label;
    }

    /**
     * Imposta il prefisso per l'ancora della linguetta della tab
     *
     * @param $prefix
     */
    public function setPrefix($prefix)
    {
        $this->prefix = $prefix;
    }

    /**
     * Restituisce il prefisso per l'ancora/id della tab (default: tab)
     *
     * @return string
     */
    public function getPrefix()
    {
        return $this->prefix;
    }

    /**
     * Imposta il link per la linguetta della tab
     *
     * @param string $link
     */
    public function setLink($link)
    {
        $this->link = $link;
    }

    /**
     * Restituisce il link impostato per la tab
     *
     * @return string
     */
    public function getLink()
    {
        if (!$this->link) {
            return '#' . $this->getPrefix() . '_' . $this->getTabName();
        }

        return $this->link;
    }

    /**
     * Restituisce l'ancora per la linguetta della tab
     *
     * @deprecated
     * @return string
     */
    public function getTabAnchor()
    {
        return $this->getLink();
    }

    /**
     * Restituisce il contenuto della tab
     *
     * @return string
     */
    public function getContent()
    {
        return $this->content;
    }

    /**
     * Imposta il contenuto della tab
     *
     * @param string $content
     */
    public function setContent($content)
    {
        $this->content = $content;
    }

    /**
     * Imposta lo stato attivo della tab
     *
     * @param boolean $active
     */
    public function setActive($active = true)
    {
        $this->active = $active;
    }

    /**
     * Restituisce lo stato attivo della tab
     *
     * @return boolean
     */
    public function isActive()
    {
        return $this->active;
    }

    /**
     * Indica se la tab è un link ad un'altra pagina o linka un contenuto nella pagina corrente
     *
     * @return bool
     */
    public function isExternalLink()
    {
        return ($this->content === null);
    }

    /**
     * Indica se la tab viene gestita tramite AJAX, caricando il contenuto del link invece di seguirlo
     *
     * @return bool
     */
    public function getIsAjax()
    {
        return $this->is_ajax;
    }

    /**
     * Renderizza il codice della tab (la linguetta per intenderci)
     *
     * @return string
     */
    public function renderTab()
    {
        $active = $this->isActive() ? ' class="active"' : '';

        $data_behavior = '';

        if ($this->getIsAjax()) {
            $data_behavior = ' data-behavior="ajax"';
        } elseif ($this->isExternalLink()) {
            $data_behavior = ' data-behavior="external"';
        }

        return sprintf(
            '<li%s data-name="%s"><a href="%s"%s>%s</a></li>',
            $active,
            $this->getTabName(),
            $this->getLink(),
            $data_behavior,
            $this->getTabLabel()
        );
    }

    /**
     * Renderizza un pannello indipendente
     *
     * @return string
     */
    public function render()
    {
        return sprintf(
            '<div class="panel" id="%s">%s</div>',
            $this->getPrefix() . '_' . $this->getTabName(),
            $this->getContent()
        );
    }

    public function __toString(): string
    {
        return $this->render();
    }
}
