<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Admin
 */

namespace Application\Admin\Tests\Utilities;

use Application\Admin\Utilities\FilterConverter;

class FilterConverterTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var FilterConverter
     */
    protected $filter_composer;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        $this->filter_composer = new FilterConverter();
    }

    /**
     * @return array
     */
    public function conditionsProvider()
    {
        return [
            '= 0 (numero)'                                  => [
                ['EQ' => 0],
                ['`col` = :col', 'col' => 0],
            ],
            '= 1 (numero)'                                  => [
                ['EQ' => 1],
                ['`col` = :col', 'col' => 1],
            ],
            '= 0 (stringa)'                                 => [
                ['EQ' => '0'],
                ['`col` = :col', 'col' => 0],
            ],
            '= 1 (stringa)'                                 => [
                ['EQ' => '1'],
                ['`col` = :col', 'col' => 1],
            ],
            '= true'                                        => [
                ['EQ' => true],
                ['`col` = :col', 'col' => true],
            ],
            '= false'                                       => [
                ['EQ' => false],
                ['`col` = :col', 'col' => false],
            ],
            '= stringa vuota / null'                        => [
                ['EQ' => ''],
                ["(`col` = '' OR `col` IS NULL)"],
            ],
            '= campo in join'                               => [
                ['EQ' => 'bar'],
                ['foo.`col` = :col', 'col' => 'bar'],
                'foo.`col`',
            ],
            'LIKE'                                          => [
                ['LIKE' => 'val'],
                ['pongho_like(`col`, :col)', 'col' => '%val%'],
            ],
            'START'                                         => [
                ['START' => 'val'],
                ['pongho_like(`col`, :col)', 'col' => 'val%'],
            ],
            'END'                                           => [
                ['END' => 'val'],
                ['pongho_like(`col`, :col)', 'col' => '%val'],
            ],
            'IN'                                            => [
                ['IN' => [1, 2, 3]],
                ["`col` IN (1, 2, 3)"],
            ],
            'NOTIN'                                         => [
                ['NOTIN' => [1, 2, 3]],
                ["`col` NOT IN (1, 2, 3)"],
            ],
            '<'                                             => [
                ['LT' => 0],
                ['`col` < :col', 'col' => 0],
            ],
            '<='                                            => [
                ['LTE' => 0],
                ['`col` <= :col', 'col' => 0],
            ],
            '>'                                             => [
                ['GT' => 0],
                ['`col` > :col', 'col' => 0],
            ],
            '>='                                            => [
                ['GTE' => 0],
                ['`col` >= :col', 'col' => 0],
            ],
            '<>'                                            => [
                ['NOT' => 0],
                ['`col` <> :col', 'col' => 0],
            ],
            'NULL'                                          => [
                ['NULL' => 0],
                ['`col` IS NULL'],
            ],
            'NOTNULL'                                       => [
                ['NOTNULL' => 0],
                ['`col` IS NOT NULL'],
            ],
            'BETWEEN'                                       => [
                ['BETWEEN' => json_encode([1, 10])],
                ['`col` BETWEEN :col_l AND :col_r', 'col_l' => 1, 'col_r' => 10],
            ],
            'DATE = mezzanotte'                             => [
                ['DATEEQ' => '2014-01-31 00:00:00'],
                ['`col` = :col', 'col' => '2014-01-31 00:00:00'],
            ],
            'DATE = ora esatta'                             => [
                ['DATEEQ' => '2014-01-31 12:34:56'],
                ['`col` = :col', 'col' => '2014-01-31 12:34:56'],
            ],
            'DATE = tutto il giorno'                        => [
                ['DATEEQ' => '2014-01-31'],
                [
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 00:00:00',
                    'col_r' => '2014-01-31 23:59:59',
                ],
            ],
            'DATEBETWEEN intervallo in giorni'              => [
                ['DATEBETWEEN' => ['2014-01-31', '2014-02-28']],
                [
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 00:00:00',
                    'col_r' => '2014-02-28 23:59:59',
                ],
            ],
            'DATEBETWEEN intervallo esatto mezzanotte'      => [
                [
                    'DATEBETWEEN' => [
                        '2014-01-31 00:00:00',
                        '2014-02-28 00:00:00',
                    ],
                ],
                [
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 00:00:00',
                    'col_r' => '2014-02-28 00:00:00',
                ],
            ],
            'DATEBETWEEN intervallo esatto'                 => [
                [
                    'DATEBETWEEN' => [
                        '2014-01-31 12:34:56',
                        '2014-02-28 12:34:56',
                    ],
                ],
                [
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 12:34:56',
                    'col_r' => '2014-02-28 12:34:56',
                ],
            ],
            'DATEBETWEEN intervallo misto senza ora fine'   => [
                [
                    'DATEBETWEEN' => [
                        '2014-01-31 12:34:56',
                        '2014-02-28',
                    ],
                ],
                [
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 12:34:56',
                    'col_r' => '2014-02-28 23:59:59',
                ],
            ],
            'DATEBETWEEN intervallo misto senza ora inizio' => [
                [
                    'DATEBETWEEN' => [
                        '2014-01-31',
                        '2014-02-28 12:34:56',
                    ],
                ],
                [
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 00:00:00',
                    'col_r' => '2014-02-28 12:34:56',
                ],
            ],
        ];
    }

    /**
     * @dataProvider conditionsProvider
     * @param        $data
     * @param string $column_alias
     */
    public function testPrepareCondition($data, array $expected_condition, $column_alias = null): void
    {
        $condition = $this->filter_composer->toCondition('col', $data, $column_alias);

        $this->assertEquals($expected_condition, $condition);
    }
}
