<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili, Daniele Termini
 * @package Application\Admin
 */


namespace Application\Admin\Model;

use ActiveRecord\Base;

/**
 * @property integer $id
 * @property integer $parent_id
 * @property integer $position
 * @property integer $module_id
 * @property integer $permit_id
 * @property string  $name
 * @property string  $path
 * @property string  $controller
 * @property string  $css_class
 */
class AdminModule extends Base
{
    /**
     * Tabella
     */
    public static $table_name = 'admin_modules';

    /**
     * Relazioni
     */
    public static $belongs_to = [
        ['permit', 'model' => \Application\Core\Model\Permit::class],
    ];

    /**
     * Restituisce tutti gli admin module con un determinato nome
     *
     * @param $name
     * @return \Application\Admin\Model\AdminModule[]
     */
    public static function allByName($name)
    {
        return self::all(['conditions' => ['name = :name', 'name' => $name]]);
    }

    /**
     * @param string      $name
     * @param string|null $panel
     * @return \Application\Admin\Model\AdminModule
     */
    public static function findByName($name, $panel = null)
    {
        $options = [
            'select'     => '`from`.*',
            'joins'      => 'LEFT JOIN ' . self::tableName() . ' AS p ON p.id = `from`.parent_id',
            'conditions' => [
                '`from`.name = :name',
                'name' => $name,
            ],
        ];

        if ($panel) {
            $options = self::addCondition(
                $options,
                [
                    'p.name = :panel',
                    'panel' => $panel,
                ]
            );
        }

        return self::first($options);
    }

    /**
     * Restituisce tutti gli AdminModule che usano un determinato controller
     *
     * @param string $controller
     * @return \Application\Admin\Model\AdminModule[]
     */
    public static function allByController($controller)
    {
        return self::all(['conditions' => ['controller = :controller', 'controller' => $controller]]);
    }

    /**
     * Rinomina uno specifico controller degli AdminModule che ne fanno uso
     *
     * @param string $old_name
     * @param string $new_name
     */
    public static function renameController($old_name, $new_name)
    {
        /** @var \Application\Admin\Model\AdminModule $item */
        foreach (self::allByController($old_name) as $item) {
            $item->controller = $new_name;
            $item->save();
        }
    }

    /**
     * Consente di rinominare più controller degli AdminModule in una volta sola
     * passando un array di coppie old_name => new_name
     *
     * @param string[] $controllers
     */
    public static function renameControllers(array $controllers)
    {
        foreach ($controllers as $old_name => $new_name) {
            self::renameController($old_name, $new_name);
        }
    }

    /**
     * Elimina tutti gli AdminModule che fanno uso di uno specifico controller
     *
     * @param string $controller
     * @return bool
     */
    public static function deleteByController($controller)
    {
        foreach (self::allByController($controller) as $item) {
            if (!$item->delete()) {
                return false;
            }
        }

        return true;
    }

    /**
     * Restituisce un pannello
     *
     * @param $name
     * @return \Application\Admin\Model\AdminModule
     */
    public static function findPanel($name)
    {
        return self::first(['conditions' => ['name = :name AND parent_id IS NULL', 'name' => $name]]);
    }

    /**
     * Restituisce i figli di un pannello
     *
     * @return \Application\Admin\Model\AdminModule[]
     * @throws \BadMethodCallException
     */
    public function children()
    {
        if ($this->parent_id !== null) {
            throw new \BadMethodCallException('This is not a parent item.');
        }

        return self::all(['conditions' => ['parent_id = :parent', 'parent' => $this->id]]);
    }
}
