<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Admin
 */

namespace Application\Admin\Form;

use Application\Admin\Form\Field\ActionsField;
use Application\Admin\Form\Repeater\RowContent;
use Application\Admin\Form\Repeater\RowMain;
use Pongho\Core\LocalizationInterface;
use Pongho\Form\Fieldset;
use Pongho\Form\Subject\SubjectInterface;
use Pongho\Form\Utilities\FormConfig as BaseConfig;

class FormConfig extends BaseConfig
{
    /**
     * @var LocalizationInterface
     */
    protected $lang;

    /**
     * @param string                $name
     */
    public function __construct($name, SubjectInterface $subject, LocalizationInterface $lang)
    {
        parent::__construct($name, $subject);

        $this->lang = $lang;
    }

    /**
     * Crea la struttura di base delle Form di amministrazione
     *
     * @param string $active_tab
     * @param string $tab_url
     */
    public function addBaseStructure($active_tab, $tab_url)
    {
        $this->addFields(
            '',
            [
                'content' => [
                    'class'    => PanelsManagerFieldset::class,
                    'settings' => [
                        'tab_active' => $active_tab,
                        'tab_url'    => $tab_url,
                    ],
                ],
                'actions' => [
                    'class'      => ActionsFieldset::class,
                    'attributes' => ['class' => 'footer-actions'],
                ],
            ]
        );

        $this->addField(
            'actions/submit',
            [
                'class'      => ActionsField::class,
                'label'      => $this->lang->get('save'),
                'attributes' => ['class' => 'primary'],
                'settings'   => ['type' => 'submit'],
            ]
        );
    }

    /**
     * Crea la struttura con sidebar delle Form di amministrazione
     */
    public function addSidebarStructure()
    {
        $this->addBaseStructure('wrapper', '');
        $this->addTab('content/wrapper', '');

        $this->addFields(
            'content/wrapper',
            [
                'container' => [
                    'class' => ContainerFieldset::class,
                ],
                'sidebar'   => [
                    'class'      => ActionsFieldset::class,
                    'attributes' => [
                        'class' => 'sidebar',
                    ],
                ],
            ]
        );
    }

    /**
     * Funzione di supporto per addPanel*()
     *
     * @param string $label
     * @param bool   $is_accordion
     * @return array
     */
    protected function panel($label = null, $is_accordion = false, array $settings = [])
    {
        // L'accordion necessita sempre una label visibile, mentre se non specifico una label la nascondo di default
        if ($label === null && !$is_accordion) {
            $settings['hide_label'] = true;
        }

        $class = Fieldset::class;
        if (isset($settings['class'])) {
            $class = $settings['class'];
            unset($settings['class']);
        }

        $description = '';
        if (array_key_exists('description', $settings)) {
            $description = $settings['description'];
            unset($settings['description']);
        }

        $attributes = [];
        if (isset($settings['attributes'])) {
            $attributes = $settings['attributes'];
            unset($settings['attributes']);
        }

        $cssClass = [];
        if (isset($attributes['class'])) {
            $cssClass = explode(' ', $attributes['class']);
        }

        $cssClass[] = 'panel';

        if ($is_accordion) {
            $cssClass[] = 'panel-accordion';
        }

        $attributes['class'] = implode(' ', array_unique($cssClass));

        return [
            'class'       => $class,
            'label'       => $label,
            'attributes'  => $attributes,
            'settings'    => $settings,
            'description' => $description,
        ];
    }

    /**
     * Aggiunge un pannello al percorso specificato
     *
     * @param string $path
     * @param string $label
     * @param bool   $is_accordion
     */
    public function addPanel($path, $label = null, $is_accordion = false, array $settings = [])
    {
        $this->addField($path, $this->panel($label, $is_accordion, $settings));
    }

    /**
     * Aggiunge un pannello prima
     *
     * @param string $path
     * @param string $field
     * @param string $label
     * @param bool   $is_accordion
     */
    public function addPanelBefore($path, $field, $label = null, $is_accordion = false, array $settings = [])
    {
        $this->addFieldBefore($path, $field, $this->panel($label, $is_accordion, $settings));
    }

    /**
     * Aggiunge un pannello dopo
     *
     * @param string $path
     * @param string $field
     * @param string $label
     * @param bool   $is_accordion
     */
    public function addPanelAfter($path, $field, $label = null, $is_accordion = false, array $settings = [])
    {
        $this->addFieldAfter($path, $field, $this->panel($label, $is_accordion, $settings));
    }

    /**
     * Funzione di supporto per addTab*()
     *
     * @param string $label
     * @return array
     */
    protected function tab($label, array $settings = [])
    {
        return [
            'class'    => PanelFieldset::class,
            'label'    => $label,
            'settings' => $settings,
        ];
    }

    /**
     * Aggiunge una tab
     *
     * @param string $path
     * @param string $label
     */
    public function addTab($path, $label, array $settings = [])
    {
        $this->addField($path, $this->tab($label, $settings));
    }

    /**
     * Aggiunge una tab prima
     *
     * @param string $path
     * @param string $field
     * @param string $label
     */
    public function addTabBefore($path, $field, $label, array $settings = [])
    {
        $this->addFieldBefore($path, $field, $this->tab($label, $settings));
    }

    /**
     * Aggiunge una tab dopo
     *
     * @param string $path
     * @param string $field
     * @param string $label
     */
    public function addTabAfter($path, $field, $label, array $settings = [])
    {
        $this->addFieldAfter($path, $field, $this->tab($label, $settings));
    }

    /**
     * Aggiunge la struttura di base della riga dei Repeater per l'amministrazione
     *
     * @param string $path
     */
    public function addRepeaterRow($path)
    {
        $this->addFields(
            $path,
            [
                'main'    => [
                    'class' => RowMain::class,
                ],
                'content' => [
                    'class' => RowContent::class,
                ],
            ]
        );
    }
}
