<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Admin
 */

namespace Application\Admin\DependencyInjection;

use Application\Admin\Model\Manager\ArchiveLastViewManager;
use Application\Admin\Utilities\AdminControllerHelper;
use Application\Admin\Utilities\ContentLanguage;
use Application\Admin\Utilities\FilterOperators;
use Application\Admin\Utilities\PonghoUrl;
use Application\Core\I18n\Translation\Translator;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;
use Pongho\Template\Theme;

class AdminServicesProvider implements ServiceProviderInterface
{
    public function register(Container $container): void
    {
        $container->share('admin_controller_helper', function (Container $container): AdminControllerHelper {
            return new AdminControllerHelper($container);
        });

        $container->share('pongho_url', function (Container $container): PonghoUrl {
            return new PonghoUrl(
                $container->get('site'),
                $container->get('request'),
                $container->get('content_language'),
            );
        });

        $container->share('content_language', function (Container $container): ContentLanguage {
            return new ContentLanguage(
                $container->get('site'),
                $container->get('request'),
            );
        });

        $container->share('filter_operators', function (Container $container): FilterOperators {
            $operators = new FilterOperators();

            foreach ($this->defaultFilterOperators($container->get('translator')) as $type => $operator) {
                $operators->setOperator($type, $operator);
            }

            return $operators;
        });

        $container->share('archive_last_view_manager', function (): ArchiveLastViewManager {
            return new ArchiveLastViewManager();
        });

        $container->extend('theme_view', function (Theme $theme): void {
            $theme->setOption(
                'image_sizes',
                array_merge(
                    $theme->getOption('image_sizes', []),
                    [
                        'adminfeatured'  => [260, 300, 'auto'],
                        'adminslide'     => [120, 70, 'auto'],
                        'adminpreview'   => [100, 100, 'crop'],
                        'adminproduct'   => [80, 80, 'fill'],
                        'adminbaravatar' => [20, 20, 'fill'],
                    ],
                ),
            );
        });
    }

    /**
     * @return array
     */
    protected function defaultFilterOperators(Translator $translator)
    {
        return [
            'foreign'        => [
                'EQ'      => $translator->trans('equal to'),
                'NOT'     => $translator->trans('different from'),
                'IN'      => $translator->trans('in'),
                'NOTIN'   => $translator->trans('not in'),
                'NULL'    => $translator->trans('not compiled'),
                'NOTNULL' => $translator->trans('compiled'),
            ],
            'foreign-enum'   => [
                'EQ'      => $translator->trans('equal to'),
                'NOT'     => $translator->trans('different from'),
                'IN'      => $translator->trans('in'),
                'NOTIN'   => $translator->trans('not in'),
                'NULL'    => $translator->trans('not compiled'),
                'NOTNULL' => $translator->trans('compiled'),
            ],
            'foreign-string' => [
                'EQ'    => $translator->trans('equal to'),
                'NOT'   => $translator->trans('different from'),
                'LIKE'  => $translator->trans('contains'),
                'START' => $translator->trans('begins with'),
            ],
            'string'         => [
                'EQ'    => $translator->trans('equal to'),
                'NOT'   => $translator->trans('different from'),
                'LIKE'  => $translator->trans('contains'),
                'START' => $translator->trans('begins with'),
//				'END'		=> $translator->trans('ends with'),
//				'IN'		=> $translator->trans('in'),
//				'NOTIN'		=> $translator->trans('not in'),
//				'NULL'		=> $translator->trans('not compiled'),
//				'NOTNULL'	=> $translator->trans('compiled'),
            ],
            'numeric'        => [
                'EQ'    => $translator->trans('equal to'),
                'NOT'   => $translator->trans('different from'),
                'IN'    => $translator->trans('in'),
                'NOTIN' => $translator->trans('not in'),
                'LT'    => $translator->trans('lower than'),
                'LTE'   => $translator->trans('lower than or equal to'),
                'GT'    => $translator->trans('greater than'),
                'GTE'   => $translator->trans('greater than or equal to'),
//				'NULL'		=> $translator->trans('not compiled'),
//				'NOTNULL'	=> $translator->trans('compiled'),
            ],
            'date'           => [
                'DATEEQ'      => $translator->trans('equal to'),
                'DATEBETWEEN' => $translator->trans('from / to'),
//				'NOT'		=> $translator->trans('different from'),
                'LT'          => $translator->trans('before'),
//				'LTE'		=> $translator->trans('before or equal'),
                'GT'          => $translator->trans('after'),
//				'GTE'		=> $translator->trans('after or equal'),
                'NULL'        => $translator->trans('not compiled'),
                'NOTNULL'     => $translator->trans('compiled'),
            ],
            'datetime'       => [
                'DATEEQ'      => $translator->trans('equal to'),
                'DATEBETWEEN' => $translator->trans('from / to'),
//				'NOT'		=> $translator->trans('different from'),
                'LT'          => $translator->trans('before'),
//				'LTE'		=> $translator->trans('before or equal'),
                'GT'          => $translator->trans('after'),
//				'GTE'		=> $translator->trans('after or equal'),
                'NULL'        => $translator->trans('not compiled'),
                'NOTNULL'     => $translator->trans('compiled'),
            ],
            'time'           => [
                'DATEEQ'      => $translator->trans('equal to'),
                'DATEBETWEEN' => $translator->trans('from hour / to hour'),
//				'NOT'		=> $translator->trans('different from'),
                'LT'          => $translator->trans('before hour'),
//				'LTE'		=> $translator->trans('before or equal'),
                'GT'          => $translator->trans('after hour'),
//				'GTE'		=> $translator->trans('after or equal'),
                'NULL'        => $translator->trans('not compiled'),
                'NOTNULL'     => $translator->trans('compiled'),
            ],
            'enum'           => [
                'EQ' => $translator->trans('equal to'),
//				'NOT'		=> $translator->trans('different from'),
//				'NULL'		=> $translator->trans('not compiled'),
//				'NOTNULL'	=> $translator->trans('compiled'),
            ],
        ];
    }
}
