<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Pongho\Sdk\GoogleAnalytics
 */

namespace Application\Admin\Dashboard;

use Google_Client;
use Google_Service_Analytics;
use Pongho\Caching\AbstractCache;

final class GoogleAnalyticsAccessToken
{
    /**
     * Percorso del file JSON con le credenziali dell’account di servizio.
     *
     * - Accedere nella Console API di Google (https://console.developers.google.com/apis/dashboard)
     * - Creare un progetto (o selezionarne uno di esistente)
     * - Dal menù di navigazione (l’hamburger menù in alto a sinistra): IAM e amministrazione / Account di servizio
     * - Creare un nuovo account di servizio
     * - Creare una chiave e scaricare il file JSON
     *
     * Alla `$credentialsPath` va associato il percorso del file JSON.
     *
     * @var string
     */
    private $credentialsPath;

    /**
     * @var AbstractCache
     */
    private $cache;

    /**
     * @param string        $credentialsPath
     */
    public function __construct($credentialsPath, AbstractCache $cache)
    {
        $this->credentialsPath = $credentialsPath;
        $this->cache = $cache;
    }

    /**
     * @return string
     */
    public function getToken()
    {
        $cacheKey = 'google-analytics-token';
        if ($this->cache->exists($cacheKey)) {
            return $this->cache->get($cacheKey);
        }

        putenv('GOOGLE_APPLICATION_CREDENTIALS=' . $this->credentialsPath);

        $client = new Google_Client();
        $client->setApplicationName('Pongho Dashboard');
        $client->useApplicationDefaultCredentials();
        $client->setScopes(Google_Service_Analytics::ANALYTICS_READONLY);
        $client->fetchAccessTokenWithAssertion();

        $accessToken = $client->getAccessToken();

        $this->cache->save($cacheKey, $accessToken['access_token'], $accessToken['expires_in']);

        return (string) $accessToken['access_token'];
    }
}
