<?php

/*
 * This file is part of the Meta Line CookieBar package.
 *
 * (c) Web Agency Meta Line S.r.l.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace MetaLine\GDPR;

class ParserRegex implements ParserInterface
{
    public function parse(string $content, Collection $iframeWhiteList, Collection $javascriptBlackList): string
    {
        $content = preg_replace_callback(
            '#<iframe(.*?)>#s',
            function (array $matches) use ($iframeWhiteList) {
                $attributes = $matches[1];

                $src = $this->extractSrc($attributes);

                if ($src && $iframeWhiteList->isPresent($src)) {
                    return $matches[0];
                }

                $attributes = str_replace(" src=\"$src\"", " data-src=\"$src\"", $attributes);
                $attributes = $this->addClass($attributes, '_gdpr_iframe');

                return "<iframe{$attributes}>";
            },
            $content
        );

        $content = preg_replace_callback(
            '#<script(.*?)>(.*?)</script>#s',
            function (array $matches) use ($javascriptBlackList) {
                $attributes = $matches[1];

                $src = $this->extractSrc($attributes);

                if ($src && $javascriptBlackList->isPresent($src)) {
                    $attributes = str_replace(" src=\"{$src}\"", " data-src=\"{$src}\"", $attributes);
                    $attributes = $this->addClass($attributes, '_gdpr_script_src');
                    $attributes = $this->addType($attributes, 'text/plain');

                    return "<script{$attributes}></script>";
                }

                if (isset($matches[2]) && $javascriptBlackList->isPresent($matches[2])) {
                    $attributes = $this->addClass($attributes, '_gdpr_script_inline');
                    $attributes = $this->addType($attributes, 'text/plain');

                    return "<script{$attributes}>" . $matches[2] . '</script>';
                }

                return $matches[0];
            },
            $content
        );

        return $content;
    }

    private function extractSrc(string $attributes): ?string
    {
        $regex = '# src="(.*?)"#s';
        $matches = [];
        preg_match($regex, $attributes, $matches);

        if (isset($matches[1])) {
            return $matches[1];
        }

        return null;
    }

    private function addClass(string $attributes, string $class): string
    {
        $regex = '# class="(.*?)"#s';
        $matches = [];
        preg_match($regex, $attributes, $matches);

        if (isset($matches[0])) {
            return preg_replace_callback(
                $regex,
                function (array $classMatches) use ($class) {
                    if (isset($classMatches[1]) && !strpos(' ' . $classMatches[1] . ' ', ' ' . $class . ' ')) {
                        return ' class="' . $classMatches[1] . ' ' . $class . '"';
                    }

                    return $classMatches[0];
                },
                $attributes
            );
        }

        return $attributes . ' class="' . $class . '"';
    }

    private function addType(string $attributes, string $type): string
    {
        $regex = '# type="(.*?)"#s';
        $matches = [];
        preg_match($regex, $attributes, $matches);

        if (isset($matches[1])) {
            return str_replace(' type="' . $matches[1] . '"', ' type="' . $type .'"', $attributes);
        }

        return $attributes . ' type="' . $type .'"';
    }
}
