<?php

/*
 * This file is part of the Meta Line CookieBar package.
 *
 * (c) Web Agency Meta Line S.r.l.
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace MetaLine\GDPR;

use DOMDocument;
use DOMElement;

class ParserDom implements ParserInterface
{
    public function parse(string $content, Collection $iframeWhiteList, Collection $javascriptBlackList): string
    {
        if (empty($content)) {
            return '';
        }

        $dom = new DOMDocument();

        libxml_use_internal_errors(true);
        $loaded = $dom->loadHTML($content, LIBXML_SCHEMA_CREATE);

        if (!$loaded) {
            foreach (libxml_get_errors() as $error) {
                // Ignore 801 error (unknown tag)
                if (801 === $error->code) {
                    continue;
                }

                $level = LIBXML_ERR_WARNING === $error->level ? E_USER_WARNING : E_USER_ERROR;
                trigger_error($error->message, $level);
            }

            libxml_clear_errors();
        }

        libxml_use_internal_errors(false);

        /** @var DOMElement $iframe */
        foreach ($dom->getElementsByTagName('iframe') as $iframe) {
            $this->parseIframe($iframe, $iframeWhiteList);
        }

        /** @var DOMElement $script */
        foreach ($dom->getElementsByTagName('script') as $script) {
            $this->parseScript($script, $javascriptBlackList);
        }

        return mb_convert_encoding($dom->saveHTML(), 'UTF-8', 'HTML-ENTITIES');
    }

    protected function parseIframe(DOMElement $iframe, Collection $iframeWhiteList)
    {
        $src = $iframe->getAttribute('src');

        if ($src && $iframeWhiteList->isPresent($src)) {
            return;
        }

        $iframe->setAttribute('data-src', $src);
        $iframe->removeAttribute('src');
        $this->addClass($iframe, '_gdpr_iframe');
    }

    protected function parseScript(DOMElement $script, Collection $javascriptBlackList)
    {
        $src = $script->getAttribute('src');

        if ($src && $javascriptBlackList->isPresent($src)) {
            $script->setAttribute('type', 'text/plain');
            $script->setAttribute('data-src', $src);
            $script->removeAttribute('src');
            $this->addClass($script, '_gdpr_script_src');
        } elseif ($script->nodeValue && $javascriptBlackList->isPresent($script->nodeValue)) {
            $script->setAttribute('type', 'text/plain');
            $this->addClass($script, '_gdpr_script_inline');
        }
    }

    protected function addClass(DOMElement $element, string $className)
    {
        $cssClass = trim($element->getAttribute('class'));

        if ($cssClass === '') {
            $element->setAttribute('class', $className);
        } elseif (strpos($cssClass, $className) === false) {
            $element->setAttribute('class', $cssClass . ' ' . $className);
        }
    }
}
